package org.simantics.databoard.example.old;
import java.io.IOException;
import java.net.Inet4Address;
import java.util.HashMap;
import java.util.Map;

import org.simantics.databoard.Methods;
import org.simantics.databoard.annotations.Unit;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.method.Client;
import org.simantics.databoard.method.MethodInterface;
import org.simantics.databoard.method.Server;
import org.simantics.databoard.serialization.SerializationException;

/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/

public class RPCExample3 {

	public interface SiCarProvider {
		SiCar findCar(String modelName) throws CarNotFoundError;
	}
	
	static public class SiCar {
		public String modelName;		
		public @Unit("km/h") double maxVelocity;		
		public @Unit("kg") double mass;		
		public @Unit("cm") double length; 
		public @Unit("kW") double power;
		
		public SiCar(String modelName, double maxVelocity, double mass, double length, double power) 
		{
			this.modelName = modelName;
			this.maxVelocity = maxVelocity;
			this.mass = mass;
			this.length = length;
			this.power = power;
		}
		
		@Override
		public String toString() {
			return String.format("Name:%s, Mass: %2.0f kg, Power: %2.0f kW, Speed: %2.0f km/h, Length: %2.0f cm", modelName, mass, power, maxVelocity, length);
		}
	}

	
	public interface ImCarProvider {
		ImCar findCar(String modelName) throws CarNotFoundError;
	}	

	static public class ImCar {
		public String modelName;		
		public @Unit("mph") float maxVelocity;		
		public @Unit("lbs") float mass;		
		public @Unit("ft") float length; 
		public @Unit("hp(M)") float power;
		
		public ImCar(String modelName, float maxVelocity, float mass, float length, float power) 
		{
			this.modelName = modelName;
			this.maxVelocity = maxVelocity;
			this.mass = mass;
			this.length = length;
			this.power = power;
		}
		
		@Override
		public String toString() {
			return String.format("Name:%s, Mass: %2.0f lbs, Power: %2.0f hp, Speed: %2.0f mph, Length: %2.0f ft", modelName, mass, power, maxVelocity, length);
		}
	}
	
	
	// Implementation
	public static class CarCatalog implements SiCarProvider {
		Map<String, SiCar> cars = new HashMap<String, SiCar>();
		@Override
		public SiCar findCar(String modelName) 
		throws CarNotFoundError{
			if (cars.containsKey(modelName)) return cars.get(modelName);
			throw new CarNotFoundError(modelName);
		}
		
		public void addCar(SiCar car) {
			cars.put(car.modelName, car);
		}

	}
	
	// Example Code 	
	public static void main(String[] args) throws IOException, CarNotFoundError, BindingConstructionException, SerializationException, BindingException {
		// Server
		CarCatalog catalog = new CarCatalog();
		catalog.addCar( new SiCar("Fiat 500", 205., 1034., 355., 75.) );
		
		MethodInterface serverMi = Methods.bindInterface(SiCarProvider.class, catalog);
		Server myServerSocket = new Server(8193, serverMi);
		
		// Client
		Client myClientSocket = new Client(Inet4Address.getByName("localhost"), 8193);
		
		try {
			SiCarProvider siCarProvider = Methods.createProxy(SiCarProvider.class, serverMi);
			ImCarProvider imCarProvider = Methods.createProxy(ImCarProvider.class, Methods.adaptMethods(serverMi, Methods.getInterfaceType(ImCarProvider.class).getMethodDefinitions()));
			
			System.out.println( siCarProvider.findCar("Fiat 500") );
			System.out.println( imCarProvider.findCar("Fiat 500") );
			
		} finally {
			myClientSocket.close();
			myServerSocket.close();
		}
	}	

	public static class CarNotFoundError extends Exception {
		public String modelName;
		public CarNotFoundError(String modelName) {
			this.modelName = modelName;
		}
		@Override
		public String toString() {
			return modelName+" is not in the catalog";
		}
	}
	
	
}

