package org.simantics.databoard.example.old;
import java.util.Arrays;

import org.simantics.databoard.Methods;
import org.simantics.databoard.method.Interface;
import org.simantics.databoard.method.MethodInterface;
import org.simantics.databoard.method.MethodInterface.AsyncResult;
import org.simantics.databoard.method.MethodInterface.Method;
import org.simantics.databoard.method.MethodTypeBinding;

/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/

/**
 * In this example we bind an instance (myProcessor) with to a InterfaceType (commandProcessorInterface),
 * to create InterfaceBinding (myProcessorBinding).
 * 
 * myProcessor is accessed 2 ways:
 *  1. acquiring a method access from myProcessorBinding and invoking that 
 *  2. creating a proxy implementation to ICommandProcessor and invoke that
 * 
 */
public class MethodInterfaceExample {

	public interface ICommandProcessor {
		String execute(String command, int...args) throws Error1;
	}

	// Description of the ICommandProcessor interface
	public final static Interface commandProcessorInterface = Methods.getInterfaceTypeUnchecked(ICommandProcessor.class);
	

	public static class Error1 extends Exception {
		
		private static final long serialVersionUID = 1L;
		
		public String msg;
		public Error1(String msg) {
			this.msg = msg;
		}
		@Override
		public String toString() {
			return msg;
		}
	}
	
	public static void main(String[] args) throws Exception {

		////// Server //////
		// Create service handler
		ICommandProcessor myProcessor = new ICommandProcessor() {
			@Override			
			public String execute(String command, int...args) throws Error1 {
				if (!command.equals("start")) throw new Error1("Unknown command "+command);
				return "Program started "+command+" with args "+Arrays.toString(args);
			}					
		};		

		////// Create Binding //////		
		// Adapt the myProcessor to an InterfaceBinding
		MethodInterface myProcessorBinding = Methods.bindInterface(commandProcessorInterface, myProcessor);
		
		
		////// Invoke Binding //////
		{
			// Get method access
			MethodTypeBinding b = Methods.getMethodTypeBinding( ICommandProcessor.class.getMethod("execute", String.class, int[].class) );
			Method m = myProcessorBinding.getMethod( b );

			// Invoke the method access (similiar to reflection)
			AsyncResult res = m.invoke( new Object[] { "start", new int[] {666, 777} } );
			System.out.println( res.waitForResponse() );			
		}
		
		////// Invoke Proxy //////
		{
			// Create proxy implementation that implements CommandProcessor and invokes myProcessorbinding
			ICommandProcessor myProxy = Methods.createProxy(ICommandProcessor.class, myProcessorBinding);
		
			// Execute command "start"
			try {
				String result = myProxy.execute("start", 5, 6, 72, 7423);
				System.out.println(result);
			} catch (Error1 e) {
				System.err.println(e);
			}

			// Execute command "fault"
			try {
				String result = myProxy.execute("fault");
				System.out.println(result);
			} catch (Error1 e) {
				System.err.println(e);
			}
		}

		
	}
	
}

