/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.example;

import java.awt.geom.Rectangle2D;
import java.util.Arrays;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.MutableVariant;

public class StringVariantBindingExample {

	public static void main(String[] args) throws Exception {

		//
		// You can escape any string to a string that is filename and URL compatible,
		// using StringVariantBinding
		//		
		String str = "Hello World";
		String escapedString = (String) Bindings.STR_VARIANT.create(Bindings.STRING, str);
		String unescapedString = (String) Bindings.STR_VARIANT.getContent(escapedString, Bindings.STRING);
		System.out.println("Unescaped String: "+unescapedString);
		System.out.println("Escaped String: "+escapedString);
		System.out.println();
		
		str = "http://www.simantics.org/path?query=value";
		escapedString = (String) Bindings.STR_VARIANT.create(Bindings.STRING, str);
		unescapedString = (String) Bindings.STR_VARIANT.getContent(escapedString, Bindings.STRING);
		System.out.println("Unescaped String: "+unescapedString);
		System.out.println("Escaped String: "+escapedString);
		System.out.println();
		
		// 
		// The escaped string is actually a string representation of a variant.
		// It contains information of the data type as well.
		// Strings are prefixed with "S", Integers with "I", Longs "L",
		// Other types are Base64 encoded binary which are not human readable.
		//
		// 
		// This means you can convert almost any value Integer, Long, Array, int[], 
		// Map, Set, Bean-like Class, Immutable-like class to a string and back
		// to an object.
		//
		Object value = 5;
		escapedString = (String) Bindings.STR_VARIANT.create(Bindings.INTEGER, value);
		Object unescapedObject = (Integer) Bindings.STR_VARIANT.getContent(escapedString, Bindings.INTEGER);
		System.out.println("Unescaped Object: "+unescapedObject);
		System.out.println("Escaped String: "+escapedString);
		System.out.println();

		value = new int[] {1, 2, 3, 4, 5, 6};
		escapedString = (String) Bindings.STR_VARIANT.create(Bindings.INTEGER, value);
		unescapedObject = (int[]) Bindings.STR_VARIANT.getContent(escapedString, Bindings.INTEGER);		
		System.out.println("Unescaped Object: "+Arrays.toString((int[]) unescapedObject));
		System.out.println("Escaped String: "+escapedString);
		System.out.println();

		//
		// You can change the class to a compatible class with another binding
		// int[] -> Integer[]
		//
		Binding binding1 = Bindings.INT_ARRAY;
		Binding binding2 = Bindings.getBinding(Integer[].class);
		value = new int[] {1, 2, 3, 4, 5, 6};
		escapedString = (String) Bindings.STR_VARIANT.create(binding1, value);
		unescapedObject = (Integer[]) Bindings.STR_VARIANT.getContent(escapedString,  binding2);		
		System.out.println("Unescaped Object: "+Arrays.toString((Integer[]) unescapedObject));
		System.out.println("Escaped String: "+escapedString);
		System.out.println();

		//
		// Binding.getBinding creates a binding for many classes by inspecting the class using a Reflection
		//
		Binding rectangleBinding = Bindings.getBinding(Rectangle2D.Double.class);
		value = new Rectangle2D.Double(10, 10, 200, 200);
		escapedString = (String) Bindings.STR_VARIANT.create(rectangleBinding, value);
		unescapedObject = (Rectangle2D) Bindings.STR_VARIANT.getContent(escapedString, rectangleBinding);		
		System.out.println("Unescaped Object: "+unescapedObject);
		System.out.println("Escaped String: "+escapedString);
		System.out.println();

		//
		// ObjectVariantBinding binds many java classes to java.lang.Object
		// You can adapt Objects, which are bound as variants, to strings, also bound as variants 
		//
		value = "Some object";
		String stringVariant = (String) Bindings.adapt(value, Bindings.OBJECT, Bindings.STR_VARIANT);
		Object objectVariant = Bindings.adapt(stringVariant, Bindings.STR_VARIANT, Bindings.OBJECT);
		MutableVariant variantVariant = (MutableVariant) Bindings.adapt(stringVariant, Bindings.STR_VARIANT, Bindings.MUTABLE_VARIANT);		
		System.out.println("Object Variant: "+objectVariant+" ("+objectVariant.getClass().getName()+")");
		System.out.println("String Variant: "+stringVariant);
		System.out.println("Variant Variant: "+variantVariant);
		System.out.println();
		
	}
	
}

