/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt.widgets;

import java.util.concurrent.CopyOnWriteArrayList;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactory;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.ui.SWTUtils;

public class Button extends WidgetImpl {

    private ReadFactory<?, String> textFactory;
    private ReadFactory<?, String> tooltipFactory;
    private ReadFactory<?, ImageDescriptor> imageFactory;
    private ReadFactory<?, Boolean> selectionFactory;
    private CopyOnWriteArrayList<SelectionListener> selectionListeners = new CopyOnWriteArrayList<SelectionListener>();

    private final org.eclipse.swt.widgets.Button button;

    public Button(Composite parent, WidgetSupport support, int style) {
    	super(support);
        button = new org.eclipse.swt.widgets.Button(parent, style);
        support.register(this);
    }

    public void setTextFactory(ReadFactory<?, String> textFactory) {
        this.textFactory = textFactory;
    }

    public void setTooltipFactory(ReadFactory<?, String> tooltipFactory) {
        this.tooltipFactory = tooltipFactory;
    }

    public void setImageFactory(ReadFactory<?, ImageDescriptor> imageFactory) {
        this.imageFactory = imageFactory;
    }

    public void setSelectionFactory(ReadFactory<?, Boolean> selectionFactory) {
        this.selectionFactory = selectionFactory;
    }

    public org.eclipse.swt.widgets.Button getWidget() {
        return button;
    }

    @Override
    public Control getControl() {
        return button;
    }

    @Override
    public void setInput(ISessionContext context, Object input) {

        if (selectionListeners != null) {
            for (SelectionListener listener : selectionListeners) {
                if(listener instanceof Widget) {
                    ((Widget) listener).setInput(context, input);
                }
            }
        }


        if(textFactory != null) {
            textFactory.listen(context, input, new Listener<String>() {

                @Override
                public void exception(final Throwable t) {
                    SWTUtils.asyncExec(button, () -> {
                        if(isDisposed()) return;
//                        System.out.println("Button received new text: " + text);
                        button.setText(t.toString());
                    });
                }

                @Override
                public void execute(final String text) {
                    SWTUtils.asyncExec(button, () -> {
                        if(isDisposed()) return;
//                        System.out.println("Button received new text: " + text);
                        button.setText(text);
                    });
                }

                @Override
                public boolean isDisposed() {
                    return button.isDisposed();
                }

            });
        }

        if(tooltipFactory != null) {
            tooltipFactory.listen(context, input, new Listener<String>() {

                @Override
                public void exception(Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }

                @Override
                public void execute(final String text) {
                    SWTUtils.asyncExec(button, () -> {
                        if(isDisposed()) return;
//                        System.out.println("Button received new tooltip: " + text);
                        button.setToolTipText(text);
                    });
                }

                @Override
                public boolean isDisposed() {
                    return button.isDisposed();
                }

            });
        }

        if(imageFactory != null) {
            imageFactory.listen(context, input, new Listener<ImageDescriptor>() {

                @Override
                public void exception(Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }

                @Override
                public void execute(final ImageDescriptor imageDescriptor) {
                    SWTUtils.asyncExec(button, () -> {
                        if(isDisposed()) return;
//                        System.out.println("Button received new image");
                        ResourceManager rm = support.getParameter(WidgetSupport.RESOURCE_MANAGER);
                        if (rm != null) {
                            Image image = (Image) rm.get(imageDescriptor);
                            button.setImage(image);
                        }
                        // TODO: how can we resize without this knife??
                        button.getParent().layout();
                        button.getParent().getParent().layout();
                    });
                }

                @Override
                public boolean isDisposed() {
                    return button.isDisposed();
                }

            });
        }

        if (selectionFactory != null) {
            selectionFactory.listen(context, input, new Listener<Boolean>() {
                @Override
                public void exception(Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }
                @Override
                public void execute(final Boolean selected) {
                    SWTUtils.asyncExec(button, () -> {
                        if(isDisposed()) return;
                        button.setSelection(Boolean.TRUE.equals(selected));
                    });
                }
                @Override
                public boolean isDisposed() {
                    return button.isDisposed();
                }
            });
        }
    }

    public void setText(String text) {
        button.setText(text);
    }

    public <T> void setText(final ParametrizedRead<T, String> read) {
        Simantics.getSession().async(new UniqueRead<String>() {
            @Override
            public String perform(ReadGraph graph) throws DatabaseException {
                T input = support.getInput(graph);
                return graph.syncRequest(read.get(input));
            }
        }, new Listener<String>() {
            @Override
            public void exception(Throwable t) {
                ErrorLogger.defaultLogError(t);
            }

            @Override
            public void execute(final String text) {
                if(isDisposed()) return;
                button.getDisplay().asyncExec(() -> {
                    if(isDisposed()) return;
                    button.setText(text);
                });
            }

            @Override
            public boolean isDisposed() {
                return button.isDisposed();
            }
        });
    }

    public void setTooltipText(String text) {
        button.setToolTipText(text);
    }

    public void setImage(Image image) {
        button.setImage(image);
    }

    public void setSelection(boolean selected) {
        button.setSelection(selected);
    }

    public synchronized void addSelectionListener(SelectionListener listener) {
        selectionListeners.add(listener);
        button.addSelectionListener(listener);
    }

}
