/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.BuiltinKeys.ViewpointKey;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;

/**
 * A very simple viewpoint that by default browses Consists Of relations and
 * accepts as objects any children that are of any of the specified accepted
 * types.
 * 
 * <p>
 * The accepted types must be returned through {@link #getAcceptedTypes()}
 * </p>
 * 
 * <p>
 * The children production can be customized by overriding
 * {@link #getPotentialChildren(ReadGraph, Resource)}. Whatever is produced by this
 * method will be filtered through the list of accepted resource types.
 * </p>
 * 
 * <p>
 * IMPORTANT: it is vital to actually implement this class and override the
 * {@link #getAcceptedTypes(ReadGraph)} method in the process. Otherwise the
 * equality comparisons of the resource queries used within LazyViewpoint will
 * not work properly!
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public abstract class LazyTypeAcceptViewpoint extends LazyViewpoint {
    
    public LazyTypeAcceptViewpoint(PrimitiveQueryUpdater updater, NodeContext context, ViewpointKey key) {
        super(updater, context, key);
    }

    protected abstract Resource[] getAcceptedTypes(ReadGraph g);
    
    protected Collection<Resource> getPotentialChildren(ReadGraph g, Resource r) throws DatabaseException {
    	Layer0 L0 = Layer0.getInstance(g);
        return g.getObjects(r, L0.ConsistsOf);
    }

    @Override
    public NodeContext[] children(ReadGraph g) throws DatabaseException {
        Resource input = getInput(Resource.class);
        Resource[] accepted = getAcceptedTypes(g);
        Collection<Resource> children = getPotentialChildren(g, input);
        ArrayList<NodeContext> resultContexts = new ArrayList<NodeContext>();
        for (Resource child : children) {
            for (Resource acc : accepted) {
                if (g.isInstanceOf(child, acc))
                    resultContexts.add(NodeContextBuilder.buildWithInput(child));
            }
        }
        return resultContexts.toArray(new NodeContext[resultContexts.size()]);
    }

    @Override
    public Boolean hasChildren(ReadGraph g) throws DatabaseException {
        return Boolean.valueOf(children(g).length > 0);
    }

    /*
    public static class Stub extends LazyTypeAcceptViewpoint {
        Resource[] acceptedTypes;
        public Stub(PrimitiveQueryUpdater updater, INodeContext context, ViewpointKey key, Resource[] acceptedTypes) {
            super(updater, context, key);
            this.acceptedTypes = acceptedTypes;
        }
        @Override
        protected Resource[] getAcceptedTypes(Graph g) {
            return acceptedTypes;
        }
    }
    */
}
