/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import org.simantics.browsing.ui.content.Labeler.Modifier;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.StringModifier;
import org.simantics.db.layer0.adapter.StringModifierFactory;

/**
 * Please implement:
 * <ul>
 * <li>{@link #doModify(String)} - perform the requested modification.</li>
 * </ul>
 * 
 * <p>
 * Other points of customization:
 * </p>
 * <ul>
 * <li>{@link #createModifierInput(String)} - constructs an input for
 * {@link org.simantics.db.layer0.adapter.Modifier#modify(WriteGraph, Object)}
 * from the specified label given by the user.
 * <li>{@link #getInitialValue(ReadGraph)} - returns the value that should be shown
 * initially when editing. The default implementation just adapts the input to
 * its String representation, but you may want to customize this.</li>
 * <li>{@link #verifyModification(String)} - allows for last chance denial of
 * the modification after the user has signalled approval of the modification.</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 * 
 * @param <T> the input class of the used
 *        {@link org.simantics.db.layer0.adapter.Modifier}
 */
public abstract class AbstractFactoryStringModifier implements Modifier {

    protected Session        session;

    protected Resource       subject;

    protected Resource       predicate;
    
    protected Resource       object;

    protected String         initialValue;

    protected StringModifierFactory modifierFactory;
    protected StringModifier        modifier;

    /**
     * If <code>non-null</code>, the modifier could not be fetched, e.g. adapted
     * from the specified INodeContext.
     */
    protected Throwable      modifierFailed;


    /**
     * @param context
     * @param session
     */
    public AbstractFactoryStringModifier(Resource subject, Resource predicate, Resource object, RequestProcessor processor) {
        this.subject = subject;
        this.predicate = predicate;
        this.object = object;
        this.session = processor.getSession();

        try {
            processor.syncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph g) throws DatabaseException {
                    initialValue = getInitialValue(g);
                    initializeModifier(g);
                }
            });
        } catch (DatabaseException e) {
            modifierFailed = e;
        }
    }

    protected void initializeModifier(ReadGraph graph) throws DatabaseException {
        modifierFactory = graph.getPossibleAdapter(subject, StringModifierFactory.class);
        if (modifierFactory != null)
            modifier = modifierFactory.createModifier(graph, predicate, object);
        if (modifier == null) {
            modifierFactory = null;
            modifier = graph.adapt(object, StringModifier.class);
        }
    }

    /**
     * @param g
     * @return the value that shall be returned by {@link #getValue()}
     */
    protected String getInitialValue(ReadGraph g) throws DatabaseException {
        return g.adapt(object, String.class);
    }

    /**
     * @return the modifier
     */
    protected StringModifier getModifier() {
        return modifier;
    }

    protected StringModifierFactory getModifierFactory() {
        return modifierFactory;
    }
    
    @Override
    public String getValue() {
        return initialValue;
    }

    @Override
    public String isValid(String label) {
        if (modifierFailed != null)
            return modifierFailed.getMessage();
        if (modifier == null)
            return "No modifier available";
        String t = createModifierInput(label);
        return modifier.isValid(t);
    }

    @Override
    public final void modify(String label) {
        if (modifierFailed != null)
            // TODO: throw exception?
            return;
        String t = createModifierInput(label);
        if (!verifyModification(t))
            return;
        doModify(t);
    }

    /**
     * Called one last time before actually performing the modifying write
     * transaction to verify whether this is really desired or not.
     * 
     * <p>
     * This default implementation will always allow the modification to proceed.
     * </p>
     * 
     * @param label the label to be given to the modifier
     * @return <code>true</code> to go forward with the transaction,
     *         <code>false</code> to bail out
     */
    protected boolean verifyModification(String label) {
        return true;
    }

    public abstract void doModify(String label);

    /**
     * Override if necessary.
     * 
     * @param label
     * @return
     */
    public String createModifierInput(String label) {
        return label;
    }

};
