package org.simantics.backup;

import java.nio.file.Path;
import java.util.concurrent.Future;

/**
 * Interface for providing backup capabilities to Simantics products.
 * BackupProviders are used in {@link BackupProviderService}.
 * 
 * @author Jani Simomaa
 */
public interface IBackupProvider {

    /**
     * Lock the resources that are going to be backed up in a way that allows
     * making an atomic and consistent copy of the backed up resources.
     */
    void lock() throws BackupException;

    /**
     * Initiates or executes the backing up procedure of this provider. The
     * backup procedure is allowed to return from this method and complete
     * asynchronously. A {@link Future} is returned to allow waiting for backup
     * to complete.
     * 
     * @param targetPath
     * @param revision
     * @return a future that can be waited upon to wait for the backup procedure
     *         to complete.
     */
    Future<BackupException> backup(Path targetPath, int revision) throws BackupException;

    /**
     * The counterpart of {@link #lock()} that must be invoked and invoked only
     * after {@link #lock()} has been successfully invoked
     */
    void unlock() throws BackupException;

    /**
     * Restore implementation for this provider.
     * 
     * @param fromPath
     * @param revision
     * 
     * TODO: change to return {@link Future} that can be waited upon
     */
    void restore(Path fromPath, int revision) throws BackupException;

}
