package org.simantics.backup.db;

import java.nio.file.Path;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;

import org.simantics.backup.BackupException;
import org.simantics.backup.IBackupProvider;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.scl.runtime.function.Function2;
import org.simantics.scl.runtime.tuple.Tuple0;

public class ModelledBackup implements IBackupProvider {

    private final Function1<Tuple0, Object> lockFunc;
    private final Function2<String, Integer, Function1<Tuple0, String>> backupFunc;
    private final Function1<Tuple0, Object> unlockFunc;
    private final Function2<String, Integer, Object> restoreFunc;

    private ModelledBackup(
            Function1<Tuple0, Object> lockFunc,
            Function2<String, Integer, Function1<Tuple0, String>> backupFunc,
            Function1<Tuple0, Object> unlockFunc,
            Function2<String, Integer, Object> restoreFunc) {
        this.lockFunc = lockFunc;
        this.backupFunc = backupFunc;
        this.unlockFunc = unlockFunc;
        this.restoreFunc = restoreFunc;
    }

    @Override
    public void lock() {
        lockFunc.apply(Tuple0.INSTANCE);
    }

    @Override
    public Future<BackupException> backup(Path targetPath, int revision) {

    	final Function1<Tuple0, String> fn = backupFunc.apply(targetPath.toString(), revision);
    	
    	final FutureTask<BackupException> futureTask = new FutureTask<BackupException>(new Callable<BackupException>() {
			@Override
			public BackupException call() throws Exception {
				try {
					String error = fn.apply(Tuple0.INSTANCE);
					if (error != null) {
						return new BackupException(error);
					} else {
						return null;
					}
				} catch (Throwable t) {
					return new BackupException(t);
				}
			}
    	});
        
    	Thread thread = new Thread(futureTask, "Backup Provider Thread " + backupFunc.toString());
    	thread.start();

    	return futureTask;
    }

    @Override
    public void unlock() {
        unlockFunc.apply(Tuple0.INSTANCE);
    }

    @Override
    public void restore(Path fromPath, int revision) {
        restoreFunc.apply(fromPath.toString(), revision);
    }

    public static IBackupProvider modelledBackup(
            Function1<Tuple0, Object> lockFunc,
            Function2<String, Integer, Function1<Tuple0, String>> backupFunc,
            Function1<Tuple0, Object> unlockFunc,
            Function2<String, Integer, Object> restoreFunc) {

        return new ModelledBackup(lockFunc, backupFunc, unlockFunc, restoreFunc);
    }
}
