/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on Jan 21, 2005
 * 
 * Copyright Toni Kalajainen
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simantics.acorn.internal;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Bijection map is a Map that has no values or keys, only 1:1 mappings
 * of values. These value/keys will be called with left and right side
 * values.
 * 
 * Each value can exist only once on a side
 * 
 * @author Toni Kalajainen
 */
public class BijectionMap<L, R> {

    /** The keys of tableLeft are left-side-values and
     * values are right-side-values */
    private final Map<L, R> tableLeft = new HashMap<L, R>();
    /** The keys of tableRight are right-side-values and
     * values on it are left-side-values */
    private final Map<R, L> tableRight = new HashMap<R, L>();

    public boolean containsLeft(L leftValue)
    {
        return tableLeft.containsKey(leftValue);
    }

    public boolean containsRight(R rightValue)
    {
        return tableRight.containsKey(rightValue);
    }

    public void map(L leftValue, R rightValue)
    {
        // Remove possible old mapping
        R oldRight = tableLeft.remove(leftValue);
        if (oldRight != null) {
            tableRight.remove(oldRight);
        } else {
            L oldLeft = tableRight.remove(rightValue);
            if (oldLeft != null) {
                tableLeft.remove(oldLeft);
            }
        }

        tableLeft.put(leftValue, rightValue);
        tableRight.put(rightValue, leftValue);
    }

    public int size()
    {
        return tableLeft.size();
    }

    public L getLeft(R rightValue) {
        return tableRight.get(rightValue);
    }

    public R getRight(L leftValue) {
        return tableLeft.get(leftValue);
    }

    public R removeWithLeft(L leftValue) {
        R rightValue = tableLeft.remove(leftValue);
        if (rightValue!=null)
            tableRight.remove(rightValue);
        return rightValue;
    }

    public L removeWithRight(R rightValue) {
        L leftValue = tableRight.remove(rightValue);
        if (leftValue!=null)
            tableLeft.remove(leftValue);
        return leftValue;
    }

    public Set<L> getLeftSet() {
        return tableLeft.keySet();
    }

    public Set<R> getRightSet() {
        return tableRight.keySet();
    }

    public void clear() {
        tableLeft.clear();
        tableRight.clear();
    }
}
