/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on Jan 21, 2005
 * 
 * Copyright Toni Kalajainen
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simantics.utils.bytes;

/**
 * Little Endian double <-> byte array conversions
 * Intel order
 *
 * @author Toni Kalajainen
 */
public class LEDouble {
	
	/**
	 * Convert double to byte array
	 * @param l double value
	 * @return byte array
	 */
	public static byte[] toBytes(double value)
	{
		byte array[] = new byte[8];
		long l = Double.doubleToRawLongBits(value);
		array[7] = (byte) (l & 0xff);
		array[6] = (byte) ((l >> 8) & 0xff);
		array[5] = (byte) ((l >> 16) & 0xff);
		array[4] = (byte) ((l >> 24) & 0xff);
		array[3] = (byte) ((l >> 32) & 0xff);
		array[2] = (byte) ((l >> 40) & 0xff);
		array[1] = (byte) ((l >> 48) & 0xff);
		array[0] = (byte) ((l >> 56) & 0xff);
		return array;
	}
	
	/**
	 * Write double value to byte array
	 * @param value the double value
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(double value, byte array[], int offset)
	{
		if (offset+8>array.length)
			throw new IndexOutOfBoundsException();		
		long l = Double.doubleToRawLongBits(value);
		array[7 + offset] = (byte) (l & 0xff);
		array[6 + offset] = (byte) (l >> 8);
		array[5 + offset] = (byte) (l >> 16);
		array[4 + offset] = (byte) (l >> 24);
		array[3 + offset] = (byte) (l >> 32);
		array[2 + offset] = (byte) (l >> 40);
		array[1 + offset] = (byte) (l >> 48);
		array[0 + offset] = (byte) (l >> 56);
	}
	
	/**
	 * Write double value to byte array
	 * @param value the double value
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(double value, byte array[])
	{
		if (array.length<8)
			throw new IndexOutOfBoundsException();		
		long l = Double.doubleToRawLongBits(value);
		array[7] = (byte) (l & 0xff);
		array[6] = (byte) (l >> 8);
		array[5] = (byte) (l >> 16);
		array[4] = (byte) (l >> 24);
		array[3] = (byte) (l >> 32);
		array[2] = (byte) (l >> 40);
		array[1] = (byte) (l >> 48);
		array[0] = (byte) (l >> 56);
	}	
	
	/**
	 * read double value from byte array
	 * @param array the array
	 * @param offset offset
	 * @return the value
	 */
	public static double toDouble(byte array[], int offset)
	{
		if (offset+8>array.length)
			throw new IndexOutOfBoundsException();		
		return 
			Double.longBitsToDouble(
			( ((long) array[7 + offset] & 0xFF) ) |
			( ((long) array[6 + offset] & 0xFF) << 8) |
			( ((long) array[5 + offset] & 0xFF) << 16) | 
			( ((long) array[4 + offset] & 0xFF) << 24) | 
			( ((long) array[3 + offset] & 0xFF) << 32) | 
			( ((long) array[2 + offset] & 0xFF) << 40) | 
			( ((long) array[1 + offset] & 0xFF) << 48) | 
			( ((long) array[0 + offset] & 0xFF) << 56)
		);
	}
	
	/**
	 * read double value from byte array
	 * @param array the array
	 * @return the value
	 */
	public static double toDouble(byte array[])
	{
		if (8>array.length)
			throw new IndexOutOfBoundsException();		

		return 
		Double.longBitsToDouble(
		( ((long) array[7] & 0xFF) ) |
		( ((long) array[6] & 0xFF) << 8) |
		( ((long) array[5] & 0xFF) << 16) | 
		( ((long) array[4] & 0xFF) << 24) | 
		( ((long) array[3] & 0xFF) << 32) | 
		( ((long) array[2] & 0xFF) << 40) | 
		( ((long) array[1] & 0xFF) << 48) | 
		( ((long) array[0] & 0xFF) << 56)
		);
	}

	/**
	 * Test cases
	 * @param args
	 */
	public static void main(String[] args) {
		System.out.println("min="+Double.MIN_VALUE+" max="+Double.MAX_VALUE);
		double value = -123.123;
		byte array[] = toBytes(value);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		write(value, array, 0);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		write(value, array);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		value = toDouble(array, 0);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
				
		value = toDouble(array);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
		
	}
	
	public static void printByteArray(byte array[]) {
		for (int i=0; i<array.length; i++) {
			System.out.print(array[i] & 0xff);
			if (i<array.length-1) 
				System.out.print(",");
		}
	}
}
