/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on Jan 21, 2005
 * 
 * Copyright Toni Kalajainen
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simantics.utils.bytes;

import org.simantics.utils.strings.EString;

/**
 * Byte array <-> string conversions
 * 
 * All conversions have length in the beginning (32-bit integer) and
 * followed by actual chars 
 * 
 * the length is in Big Endian
 *
 * @author Toni Kalajainen
 */
public class BEString {

	public static byte[] toBytes(String s) {
		int size = s.length();
		byte array[] = new byte[size + 4];
		// write length
		BEInt.write(size, array);
		// write chars
		for (int i=0; i<size; i++)
			array[i+4] = (byte) (s.charAt(i) );
		return array;
	}
	
	/**
	 * Write string to byte array
	 * @param str the string
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(String str, byte array[], int offset)
	{
		int size = str.length();
		if (offset+4+size>array.length)
			throw new IndexOutOfBoundsException();		
		BEInt.write(size, array, offset);
		for (int i=0; i<size; i++)
			array[i+4+offset] = (byte) (str.charAt(i) );			
	}
	
	/**
	 * Write string to byte array
	 * @param str the string
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(String str, byte array[])
	{
		int size = str.length();
		if (4+size>array.length)
			throw new IndexOutOfBoundsException();		
		BEInt.write(size, array);
		for (int i=0; i<size; i++)
			array[i+4] = (byte) (str.charAt(i) );			
	}
	
	public static String toString(byte array[], int offset) {
		if (offset+4>array.length)
			throw new IndexOutOfBoundsException();
		// read size
		int size = BEInt.toInt(array, offset);
		if (offset+4+size>array.length)
			throw new IndexOutOfBoundsException();
		// read chars
		//return new String(array, 4+offset, size);
		char chars[] = new char[size];
		for (int i=0; i<size; i++)
			chars[i] = (char) (array[i+4+offset] & 0xff);	
		return new String(chars);
	}
	
	public static String toString(byte array[]) {
		if (4>array.length)
			throw new IndexOutOfBoundsException();
		// read size
		int size = BEInt.toInt(array);
		if (4+size>array.length)
			throw new IndexOutOfBoundsException();
		// read chars
		//return new String(array, 4, size);
		char chars[] = new char[size];
		for (int i=0; i<size; i++)
			chars[i] = (char) (array[i+4] & 0xff);	
		return new String(chars);
	}
	
	/**
	 * Big Endian hex
	 * @param value
	 * @param decimals
	 * @return
	 */
	public static String intToHex(int value, int decimals) {
		String result="";		
		for (int i=0; i<decimals; i++) {
			result += EString.HEX_VALUES[(value>>4) & 0xF];
			result += EString.HEX_VALUES[value & 0xF];
			value = value >> 8;			
		}		
		return result;
	}	
	
	public static void main(String[] args) {
		String value = "STRING 01234"+(char)(128)+(char)(129)+(char)(255);
		
		int X = 500;
		System.out.println(X+" = "+intToHex(X, 4)); 		
		
		byte array[] = toBytes(value);			
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		write(value, array);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		write(value, array, 0);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		value = toString(array, 0);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
				
		value = toString(array);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
		
		array = toBytes(value);			
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
						
	}
	
	public static void printByteArray(byte array[]) {
		for (int i=0; i<array.length; i++) {
			System.out.print(array[i] & 0xff);
			if (i<array.length-1) 
				System.out.print(",");
		}
	}	
}
