/*******************************************************************************
 * Copyright (c) 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - (#7066) initial API and implementation
 *******************************************************************************/
package org.simantics.views.text.internal;

import org.eclipse.jface.text.Document;
import org.eclipse.mylyn.wikitext.mediawiki.MediaWikiLanguage;
import org.eclipse.mylyn.wikitext.parser.markup.MarkupLanguage;
import org.eclipse.mylyn.wikitext.ui.editor.MarkupSourceViewer;
import org.eclipse.mylyn.wikitext.ui.editor.MarkupSourceViewerConfiguration;
import org.eclipse.mylyn.wikitext.ui.editor.ShowInTargetBridge;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.services.IServiceLocator;
import org.eclipse.ui.swt.IFocusService;
import org.simantics.views.ViewUtils;
import org.simantics.views.swt.client.base.SingleSWTViewNode;

/**
 * @author Tuukka Lehtonen
 * @since 1.28.0
 */
public class SWTMarkupSourceViewer extends SingleSWTViewNode<StyledText> {

	private static final long serialVersionUID = 3034624586096417826L;

	private MarkupSourceViewer viewer;
	private boolean textInitialized = false;

	@Override
	public void createControls(Composite parent) {
		MarkupLanguage language = new MediaWikiLanguage();
		viewer = new MarkupSourceViewer(parent, null, style | SWT.WRAP, language);
		viewer.setEditable(false);
		MarkupSourceViewerConfiguration configuration = new MarkupSourceViewerConfiguration(Activator.getDefault().getPreferenceStore());
		configuration.setMarkupLanguage(language);
		configuration.setShowInTarget(new ShowInTargetBridge(viewer));
		viewer.configure(configuration);
		viewer.setDocument(new Document(text != null ? text : ""));
		control = viewer.getTextWidget();
		control.setData(TextViewerConstants.KEY_UNDO_MANAGER, viewer.getUndoManager());
		control.setEnabled(false);

		setProperties();

		// Allow undo/redo handler to be bound to this text editor's focus
		IServiceLocator locator = getSite();
		if (locator != null) {
			IFocusService focusService = locator.getService(IFocusService.class);
			if (focusService != null) {
				focusService.addFocusTracker(control, TextViewerConstants.CONTEXT_IN_TEXT_VIEWER);
			}
		}

		control.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ViewUtils.setWorkbenchSelection(viewer.getSelection());
			}
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}
		});
	}

	@Override
	public void synchronizeText(String text) {
		this.text = text;
		if (text != null) {
			// Try to keep the vertical scroll position of the text widget
			int caretOffset = control.getCaretOffset();
			int charCount = control.getCharCount();
			int topIndex = viewer.getTopIndex();
			int diff = text.length() - charCount;
			int newCaretOffset = Math.max(0, Math.min(caretOffset + diff, text.length()));

			viewer.getDocument().set(text);
			viewer.setTopIndex(topIndex);
			control.setCaretOffset(newCaretOffset);
			viewer.setEditable(true);
			control.setEnabled(true);

			// Prevent text viewer undo from allowing the
			// user to undo the text viewer back to empty.
			if (!textInitialized) {
				viewer.getUndoManager().reset();
				textInitialized = true;
			}
		} else {
			textInitialized = false;
			viewer.setEditable(false);
			control.setEnabled(false);
			viewer.getDocument().set("");
			viewer.getUndoManager().reset();
		}
	}

	public String readText() {
		return viewer.getDocument().get();
	}

	public Point readSelection() {
		return control.getSelection();
	}

}