/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui.workbench;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.IFilter;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.intro.IIntroSite;
import org.eclipse.ui.part.IPageSite;
import org.simantics.utils.ui.SWTUtils;

/**
 * Some eclipse workspace utils
 * 
 * @author Toni Kalajainen
 */
public class WorkbenchUtils {

    /**
     * Find open viewpart
     * 
     * @param primaryID primary view id
     * @param secondaryID secondary view id
     * @return view part if it exists
     */
    public static IViewPart findView(String primaryID, String secondaryID) {
        if (secondaryID==null)
            return _findView(primaryID, false);
        else
            return _findView(primaryID + ":" + secondaryID, false);
    }

    /**
     * Find open viewpart
     * 
     * @param id full id (example "primaryid:secondaryid")
     * @return <code>IViewPart</code> or null if view wasn't found
     */
    public static IViewPart findView(String id) {
        return _findView(id, false);
    }

    /**
     * Finds (and restores) all views that have a specific primary Id
     * 
     * @param primaryId primary Id
     * @return all views
     */
    public static IViewPart[] findAllViews(String primaryId) {
        List<IViewPart> result = new ArrayList<IViewPart>();
        for (IViewReference ref : getViewReferences())
            if (primaryId.equals(ref.getId()))
                result.add(ref.getView(true));
        return result.toArray(new IViewPart[0]);
    }

    /**
     * Find viewpart, opens view if view is closed
     * 
     * @param id full id
     * @return <code>IViewPart</code> or null if view wasn't found
     */
    public static IViewPart getView(String id) {
        return _findView(id, true);
    }

    public static IWorkbenchWindow getWorkbenchWindow(Shell shell)
    {
        for (IWorkbenchWindow w : PlatformUI.getWorkbench().getWorkbenchWindows())
            if (w.getShell()==shell) return w;
        return null;
    }

    /**
     * Try to get the {@link Shell} of the currently active workbench window.
     * Must be invoked from the SWT UI thread to receive results.
     * 
     * @return Shell of the currently active workbench window or
     *         <code>null</code> if there is no active window or if called from
     *         a non-UI thread
     */
    public static Shell getActiveWorkbenchWindowShell() {
        IWorkbench wb = PlatformUI.getWorkbench();
        IWorkbenchWindow window = wb.getActiveWorkbenchWindow();
        if (window == null)
            return null;
        return window.getShell();
    }

    /**
     * Get the active workbench part in currently active workbench window /
     * active page
     * 
     * @return active workbench part of the active workbench window
     */
    public static IWorkbenchPart getActiveWorkbenchPart()
    {
        IWorkbench wb = PlatformUI.getWorkbench();
        if (wb==null) return null;
        IWorkbenchWindow wbw = wb.getActiveWorkbenchWindow();
        if (wbw==null) return null;
        IWorkbenchPage wbp = wbw.getActivePage();
        if (wbp==null) return null;
        return wbp.getActivePart();
    }


    /**
     * Get the active workbench part in currently active workbench window /
     * active page
     * 
     * @return active workbench part of the active workbench window
     */
    public static IEditorPart getActiveEditor()
    {
        IWorkbench wb = PlatformUI.getWorkbench();
        if (wb==null) return null;
        IWorkbenchWindow wbw = wb.getActiveWorkbenchWindow();
        if (wbw==null) return null;
        IWorkbenchPage wbp = wbw.getActivePage();
        if (wbp==null) return null;
        return wbp.getActiveEditor();
    }

    /**
     * Get the perspective in current active workbench window / active page
     * @return Current perspective
     */
    public static IPerspectiveDescriptor getCurrentPerspective()
    {
        IWorkbench wb = PlatformUI.getWorkbench();
        if (wb==null) return null;
        IWorkbenchWindow wbw = wb.getActiveWorkbenchWindow();
        if (wbw==null) return null;
        IWorkbenchPage wbp = wbw.getActivePage();
        if (wbp==null) return null;
        return wbp.getPerspective();
    }

    /**
     * Get the perspective in current active workbench window / active page
     * @return Current perspective Id
     */
    public static String getCurrentPerspectiveId()
    {
        IPerspectiveDescriptor p = getCurrentPerspective();
        if (p==null) return null;
        return p.getId();
    }

    /**
     * Returns all <code>IViewReference</code>s
     * @return array of all <code>IViewReference</code>s
     */
    public static IViewReference[] getViewReferences() {
        Set<IViewReference> result = new HashSet<IViewReference>();

        IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
        if (windows == null)
            return null;
        for (IWorkbenchWindow wb : windows) {
            if (wb == null)
                return null;
            IWorkbenchPage pages[] = wb.getPages();
            if (pages == null)
                return null;
            for (IWorkbenchPage page : pages) {
                if (page == null)
                    continue;
                IViewReference refs[] = page.getViewReferences();
                for (IViewReference ref : refs)
                    result.add(ref);
            }
        }
        return result.toArray(new IViewReference[0]);
    }

    /**
     * Returns all <code>IViewReference</code>s for the specified workbench
     * window
     * 
     * @return array of all <code>IViewReference</code>s for the specified
     *         window
     */
    public static IViewReference[] getViewReferences(IWorkbenchWindow window) {
        Set<IViewReference> result = new HashSet<IViewReference>();
        for (IWorkbenchPage page : window.getPages()) {
            IViewReference refs[] = page.getViewReferences();
            for (IViewReference ref : refs)
                result.add(ref);
        }
        return result.toArray(new IViewReference[0]);
    }

    /**
     * Finds a <code>IViewPart</code>. Opens the view if it's closed
     * @param primaryID primary ID of the view
     * @param secondaryID secondary ID of the view or null if view has no secondary ID
     * @return <code>IViewPart</code> if view was found or opened, else returns null
     */
    public static IViewPart getView(String primaryID, String secondaryID) {
        if (secondaryID==null)
            return _findView(primaryID, true);
        return _findView(primaryID+":"+secondaryID, true);
    }

    /**
     * Finds a <code>IViewPart</code>
     * @param id id of the View
     * @param restore set to true if you want to open closed view
     * @return the <code>IViewPart</code> if view was found or restored, else returns null
     */
    private static IViewPart _findView(String id, boolean restore) {
        String primaryId = getPrimaryID(id);
        String secondaryId = getSecondaryID(id);
        IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
        if (windows == null)
            return null;
        for (IWorkbenchWindow wb : windows) {
            if (wb == null)
                return null;
            IWorkbenchPage pages[] = wb.getPages();
            if (pages == null)
                return null;
            for (IWorkbenchPage page : pages) {
                if (page == null)
                    continue;

                IViewReference vr = page.findViewReference(primaryId, secondaryId);
                if (vr == null)
                    continue;

                IViewPart vp = vr.getView(restore);
                if (vp != null)
                    return vp;
            }
        }
        return null;
    }

    /**
     * Hides a view
     * @param id the id of the view
     * @return true if view was hidden, returns false if view with id wasn't found
     */
    public static boolean hideView(String id) {
        IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
        if (windows == null)
            return false;
        for (IWorkbenchWindow wb : windows) {
            if (wb == null)
                return false;
            IWorkbenchPage pages[] = wb.getPages();
            if (pages == null)
                return false;
            for (IWorkbenchPage page : pages) {
                if (page == null)
                    continue;

                IViewReference vr = page.findViewReference(getPrimaryID(id), getSecondaryID(id));
                if (vr == null)
                    continue;
                page.hideView(vr);
                return true;

            }
        }
        return false;
    }

    /**
     * @param viewPart
     * @return
     */
    public static boolean hideView(IViewPart viewPart) {
        IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
        if (windows == null)
            return false;
        for (IWorkbenchWindow wb : windows) {
            if (wb == null)
                return false;
            IWorkbenchPage pages[] = wb.getPages();
            if (pages == null)
                return false;
            for (IWorkbenchPage page : pages) {
                if (page == null)
                    continue;
                page.hideView(viewPart);
                return true;
            }
        }
        return false;
    }

    /**
     * @param view
     * @return
     */
    public static boolean hideView(IWorkbenchWindow window, IViewPart view) {
        for (IWorkbenchPage page : window.getPages()) {
            if (page == null)
                continue;
            page.hideView(view);
            return true;
        }
        return false;
    }

    /**
     * @param view
     * @return
     */
    public static boolean hideView(IWorkbenchWindow window, IViewReference view) {
        for (IWorkbenchPage page : window.getPages()) {
            if (page == null)
                continue;
            page.hideView(view);
            return true;
        }
        return false;
    }

    /**
     * Gets the primary id from view id "prim:secon"
     * 
     * @param id full id
     * @return primary id
     */
    public static String getPrimaryID(String id) {
        String splitted[] = id.split(":");
        if (splitted == null || splitted.length < 1)
            return null;
        return splitted[0];
    }

    /**
     * Gets the secondary id from view id "prim:secon"
     * 
     * @param id full id
     * @return secondary id
     */
    public static String getSecondaryID(String id) {
        String splitted[] = id.split(":");
        if (splitted == null || splitted.length < 2)
            return null;
        return splitted[1];
    }

    /**
     * Creates the view
     * @param id the id of the view
     * @return <code>IViewPart</code> or null if view couldn't be created
     * @throws PartInitException
     */
    public static IViewPart showView(String id) throws PartInitException {
        return showView(id, IWorkbenchPage.VIEW_CREATE);
    }

    /**
     * Activates the view
     * @param id the id of the view
     * @return <code>IViewPart</code> or null if view couldn't be activated
     * @throws PartInitException
     */
    public static IViewPart activateView(String id) throws PartInitException {
        IViewPart vp = showView(id, IWorkbenchPage.VIEW_ACTIVATE);
        return vp;
    }

    /**
     * Shows the view
     * @param id the id of the view
     * @param mode <code>IWorkbenchPage.VIEW_CREATE</code> or <code>IWorkbenchPage.VIEW_ACTIVATE</code>
     * @return the <code>IViewPart</code> or null if showing the view failed
     * @throws PartInitException
     */
    public static IViewPart showView(String id, final int mode) throws PartInitException {
        // Is view already created
        IViewPart vp = findView(id);
        if (vp != null) {
            if (mode == IWorkbenchPage.VIEW_CREATE)
                return vp;

            Display display = Display.getCurrent();
            if (Thread.currentThread() == display.getThread()) {
                // This is the UI-thread
                //System.out.println("In UI thread");
                IWorkbenchPage page = vp.getViewSite().getPage();

                if (mode == IWorkbenchPage.VIEW_VISIBLE) {
                    page.bringToTop(vp);
                } else if (mode == IWorkbenchPage.VIEW_ACTIVATE) {
                    page.activate(vp);
                }
            } else {
                //System.out.println("NOT in UI thread!");
                final IViewPart fvp = vp;
                display.asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        final IWorkbenchWindow wb = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
                        IWorkbenchPage page = wb.getActivePage();
                        if (mode == IWorkbenchPage.VIEW_VISIBLE) {
                            page.bringToTop(fvp);
                        } else {
                            page.activate(fvp);
                        }
                    }
                });
            }
            return vp;
        }

        final String primaryID = getPrimaryID(id);
        final String secondaryID = getSecondaryID(id);
        if (primaryID == null/* || secondaryID == null*/)
            return null;

        // Create the view on active page
        final IWorkbenchWindow wb = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
        if (wb == null)
            return null;
        IWorkbenchPage page = wb.getActivePage();
        if (page == null) {
            IWorkbenchPage pages[] = wb.getPages();
            if (pages==null || pages.length == 0) return null;
            page = pages[0];
        }

        vp = page.showView(primaryID, secondaryID, mode);
        return vp;
    }


    /**
     * IMPORTANT: If you're using multiple workbench windows, use
     * {@link #showPerspective(IWorkbenchWindow, String, IAdaptable)} instead.
     * 
     * @param perspectiveId the ID of the perspective to activate in the
     *        currently active workbench window
     * @throws WorkbenchException if perspective activation fails
     */
    public static void showPerspective(String perspectiveId) throws WorkbenchException {
        showPerspective(PlatformUI.getWorkbench().getActiveWorkbenchWindow(), perspectiveId);
    }

    /**
     * Opens the specified perspective in the specified workbench window
     * according to the description of
     * {@link IWorkbench#showPerspective(String, IWorkbenchWindow)}.
     * 
     * <p>
     * IMPORTANT: If you're using multiple workbench windows, use
     * {@link #showPerspective(IWorkbenchWindow, String, IAdaptable)} instead.
     * </p>
     * 
     * @param window the window in which to show the specified perspective
     * @param perspectiveId the ID of the perspective to activate in the
     *        currently active workbench window
     * @throws WorkbenchException if perspective activation fails
     */
    public static void showPerspective(IWorkbenchWindow window, String perspectiveId) throws WorkbenchException {
        PlatformUI.getWorkbench().showPerspective(perspectiveId, window);
    }

    /**
     * Opens the specified perspective in the specified workbench window with
     * the specified page input according to the description of
     * {@link IWorkbench#showPerspective(String, IWorkbenchWindow, IAdaptable)}.
     * 
     * <p>
     * This is the only <code>showPerspective</code> that will force the
     * specified perspective to open in exactly the specified window. The other
     * methods cannot guarantee this. See
     * {@link IWorkbench#showPerspective(String, IWorkbenchWindow, IAdaptable)}
     * for an explanation why.
     * </p>
     * 
     * <p>
     * Example:
     * 
     * <pre>
     * WorkbenchUtils.showPerspective(window, myPerspective, window.getActivePage().getInput());
     * </pre>
     * 
     * </p>
     * 
     * @param window the window in which to show the specified perspective
     * @param perspectiveId the ID of the perspective to activate in the
     *        currently active workbench window
     * @param the page input, or <code>null</code> if there is no current input.
     *        This is used to seed the input for the page's views
     * @throws WorkbenchException if perspective activation fails
     */
    public static void showPerspective(IWorkbenchWindow window, String perspectiveId, IAdaptable input) throws WorkbenchException {
        PlatformUI.getWorkbench().showPerspective(perspectiveId, window, input);
    }

    /**
     * Close all perspectives in all open workbench windows that do not pass the
     * specified filter. The filter will get IPerspectiveDescriptor instances as
     * input.
     * 
     * @param perspectiveFilter a filter for <code>IPerspectiveDescriptor</code>s.
     */
    public static void closeFilteredPerspectives(IFilter perspectiveFilter) {
        for (IWorkbenchWindow window : PlatformUI.getWorkbench().getWorkbenchWindows()) {
            closeFilteredPerspectives(window, perspectiveFilter);
        }
    }

    /**
     * Close all perspectives in the specified workbench window that do not pass
     * the specified filter. The filter will get IPerspectiveDescriptor
     * instances as input.
     * 
     * @param perspectiveFilter a filter for <code>IPerspectiveDescriptor</code>s.
     */
    public static void closeFilteredPerspectives(IWorkbenchWindow window, IFilter perspectiveFilter) {
        for (IWorkbenchPage page : window.getPages()) {
            for (IPerspectiveDescriptor desc : page.getOpenPerspectives()) {
                if (!perspectiveFilter.select(desc)) {
                    page.closePerspective(desc, true, false);
                }
            }
        }
    }

    public static IActionBars getActionBars(IWorkbenchPart part) {
        if (part instanceof IViewPart) {
            return ((IViewPart) part).getViewSite().getActionBars();
        } else if (part instanceof IEditorPart) {
            return ((IEditorPart) part).getEditorSite().getActionBars();
        }
        throw new IllegalArgumentException("Specified IWorkbenchPart is neither IViewPart nor IEditorPart");
    }

    /**
     * @param editorId
     * @param input
     * @throws PartInitException
     *             if editor opening fails
     * @throws IllegalArgumentException
     *             if an editor (IEditorDescription) with the specified id is
     *             not found
     * @throws IllegalStateException
     *             if either current workbench window or current workbench page
     *             returns <code>null</code>.
     */
    public static IEditorPart openEditor(String editorId, IEditorInput input) throws PartInitException {
        IEditorDescriptor desc = PlatformUI.getWorkbench().getEditorRegistry().findEditor(editorId);
        if (desc == null) {
            throw new IllegalArgumentException("editor with id '" + editorId + "' not found");
        }

        IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
        if (window == null) {
            throw new IllegalStateException("active workbench window is null");
        }

        IWorkbenchPage page = window.getActivePage();
        if (page == null) {
            throw new IllegalStateException("active workbench page is null");
        }

        return page.openEditor(input, desc.getId());
    }

    /**
     * @param editorId
     * @param input
     * @param perspectiveId
     * @throws Exception
     */
    public static void openEditorInPerspective(String editorId, IEditorInput input, String perspectiveId) throws Exception {
        WorkbenchUtils.showPerspective(perspectiveId);
        openEditor(editorId, input);
    }

    /**
     * Closes an editor part.
     * 
     * @param editorPart
     *            editor part instance to close
     * @param save <code>true</code> to save changes before closing,
     *         <code>false</code> to discard any changes
     * @return <code>true</code> if the part was closed successfully
     */
    public static boolean closeAllEditors(boolean save) {
        IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
        if (windows == null)
            return false;
        for (IWorkbenchWindow wb : windows) {
            if (wb == null)
                return false;
            IWorkbenchPage pages[] = wb.getPages();
            if (pages == null)
                return false;
            for (IWorkbenchPage page : pages) {
                if (page == null)
                    continue;
                return page.closeAllEditors(save);
            }
        }
        return false;
    }

    /**
     * Closes an editor part.
     * 
     * @param editorPart
     *            editor part instance to close
     * @param save <code>true</code> to save changes before closing,
     *         <code>false</code> to discard any changes
     * @return <code>true</code> if the part was closed successfully
     */
    public static boolean closeEditor(IEditorPart editorPart, boolean save) {
        IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
        if (windows == null)
            return false;
        for (IWorkbenchWindow wb : windows) {
            if (wb == null)
                return false;
            IWorkbenchPage pages[] = wb.getPages();
            if (pages == null)
                return false;
            for (IWorkbenchPage page : pages) {
                if (page == null)
                    continue;
                return page.closeEditor(editorPart, save);
            }
        }
        return false;
    }

    /**
     * Closes all editor parts in the specified workbench window.
     * 
     * @param window editor part instance to close
     * @param save <code>true</code> to save changes before closing,
     *        <code>false</code> to discard any changes
     * @return <code>true</code> if all editor parts were closed properly,
     *         <code>false</code> otherwise
     */
    public static boolean closeEditors(IWorkbenchWindow window, boolean save) {
        boolean result = true;
        for (IWorkbenchPage page : window.getPages()) {
            if (page == null)
                continue;
            result &= page.closeAllEditors(false);
        }
        return result;
    }

    /**
     * Try to get {@link IActionBars} from the specified {@link IWorkbenchSite}.
     * There are four alternatives to what the site can be: {@link IViewSite},
     * {@link IPageSite}, {@link IEditorSite} or {@link IIntroSite}. All of them
     * have a method for retrieving {@link IActionBars}.
     * 
     * @return the action bars of the specified workbench site or
     *         <code>null</code> if the site is unrecognized or does not have an
     *         action bar
     */
    public static IActionBars getActionBars(IWorkbenchSite site) {
        if (site instanceof IViewSite)
            return ((IViewSite) site).getActionBars();
        else if (site instanceof IPageSite)
            return ((IPageSite) site).getActionBars();
        else if (site instanceof IEditorSite)
            return ((IEditorSite) site).getActionBars();
        else if (site instanceof IIntroSite)
            return ((IIntroSite) site).getActionBars();
        return null;
    }

    /**
     * Try to get {@link IStatusLineManager} from the specified
     * {@link IWorkbenchSite}.
     * 
     * @return the status line if available or <code>null</code> if not
     */
    public static IStatusLineManager getStatusLine(IWorkbenchPart part) {
        IActionBars bars = getActionBars(part);
        return bars != null ? bars.getStatusLineManager() : null;
    }

    /**
     * Try to get {@link IStatusLineManager} from the specified
     * {@link IWorkbenchSite}.
     * 
     * @return the status line if available or <code>null</code> if not
     */
    public static IStatusLineManager getStatusLine(IWorkbenchSite site) {
        IActionBars bars = getActionBars(site);
        return bars != null ? bars.getStatusLineManager() : null;
    }

    /**
     * @param partSite site of workbench part to activate
     * @return <code>false</code> if the activation was scheduled
     *         asynchronously, <code>true</code> if the part was synchronously
     *         activated
     * @throws NullPointerException
     *             if partSite is <code>null</code>
     */
    public static boolean activatePart(final IWorkbenchPartSite partSite) {
        final IWorkbenchPart part = partSite.getPart();
        IWorkbench workbench = partSite.getWorkbenchWindow().getWorkbench();
        Display display = workbench.getDisplay();
        Runnable activator = new Runnable() {
            @Override
            public void run() {
                partSite.getPage().activate(part);
            }
        };
        if (Thread.currentThread() == display.getThread()) {
            activator.run();
            return true;
        } else {
            SWTUtils.asyncExec(display, activator);
            return false;
        }
    }

}
