/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * 14.7.2006
 */
package org.simantics.utils.ui.dialogs;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

/**
 * TreeDialog is a dialog that contains a tree viewer.
 * <p>
 * Usage:
 *  setInput(<input>);
 *  setContentProvider(<cp>);
 *  setLabelProvider(<lp>);
 *  open() == Window.OK 
 * 
 * @author Toni Kalajainen
 */
public class TreeDialog extends Dialog {

    String title;    
    String message;
    Label messageLabel; 
    IInputValidator validator;
    Object initialSelection;
    Object selected[];
    TreeViewer viewer;
    ILabelProvider labelProvider;
    IStructuredContentProvider contentProvider;
    Object input;
    boolean expandTree = false;
    boolean multipleSelection = false;
    boolean useDoubleClick = true;
    
    ViewerSorter sorter;
    
    public TreeDialog(Shell parentShell, String dialogTitle,
            String dialogMessage) {
        super(parentShell);
        message = dialogMessage;
        title = dialogTitle;
    }
    
    /**
     * Set the title
     */
    protected void configureShell(Shell shell) {
        super.configureShell(shell);
        if (title != null) {
            shell.setText(title);
        }
    }
    
    /**
     * Set selection value
     */
    protected void buttonPressed(int buttonId) {
        selected = null;
        if (buttonId == IDialogConstants.OK_ID) {
            ISelection sel = viewer.getSelection();
            if (sel instanceof IStructuredSelection)
            {
                IStructuredSelection ss = (IStructuredSelection) sel;
                if (!ss.isEmpty())
                    selected = ss.toArray();
            }
        }
        super.buttonPressed(buttonId);
    }   
    
    protected Control createDialogArea(Composite parent) {        
        // create composite
        Composite composite = (Composite) super.createDialogArea(parent);
        // create message
        if (message != null) {
            messageLabel = new Label(composite, SWT.WRAP);
            messageLabel.setText(message);
            GridData data = new GridData(GridData.GRAB_HORIZONTAL
                    | GridData.GRAB_VERTICAL | GridData.HORIZONTAL_ALIGN_FILL
                    | GridData.VERTICAL_ALIGN_CENTER);
            data.widthHint = convertHorizontalDLUsToPixels(IDialogConstants.MINIMUM_MESSAGE_AREA_WIDTH);
            messageLabel.setLayoutData(data);
            messageLabel.setFont(parent.getFont());
        }
        
        assert(input!=null);
        assert(labelProvider!=null);
        assert(contentProvider!=null);
        int flags = SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.FULL_SELECTION;
        if (multipleSelection) flags |= SWT.MULTI;
        viewer = new TreeViewer(parent, flags);
        viewer.setLabelProvider(labelProvider);
        viewer.setContentProvider(contentProvider);
        viewer.setSorter(sorter);
        viewer.setInput(input);
        if (useDoubleClick)
	        viewer.addDoubleClickListener(new IDoubleClickListener() {
	            public void doubleClick(DoubleClickEvent event) {
	                //TreeDialog.this.okPressed();
	                buttonPressed(IDialogConstants.OK_ID);
	            }});
        
    
        if (expandTree)
            viewer.expandAll();              
        
        GridData gd = new GridData(GridData.GRAB_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL | 
        		                   GridData.GRAB_VERTICAL | GridData.VERTICAL_ALIGN_FILL );
        gd.heightHint = 250;
        gd.widthHint = 200;
        viewer.getTree().setLayoutData(gd);
        if (initialSelection!=null) 
            viewer.setSelection(new StructuredSelection(initialSelection));                
        applyDialogFont(composite);
        return composite;
    }

    public ILabelProvider getLabelProvider() {
        return labelProvider;
    }

    /**
     * Sets label provider
     * 
     * Must be called before dialog is opened.
     * 
     * @param labelProvider
     */
    public void setLabelProvider(ILabelProvider labelProvider) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.labelProvider = labelProvider;
    }

    public IStructuredContentProvider getContentProvider() {
        return contentProvider;
    }

    /**
     * Sets content provider
     * 
     * Must be called before dialog is opened.
     * 
     * @param contentProvider
     */
    public void setContentProvider(IStructuredContentProvider contentProvider) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.contentProvider = contentProvider;
    }

    public Object[] getSelected() {
        return selected;
    }

    public Object getSingleSelected() {
        if (selected==null || selected.length==0) return null;
        return selected[0];
    }

    public Object getInitialSelection() {
        return initialSelection;
    }

    /**
     * Sets initial selection.
     * 
     * Must be called before dialog is opened.
     * 
     * @param initialSelection
     */
    public void setInitialSelection(Object initialSelection) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.initialSelection = initialSelection;
    }

    public Object getInput() {
        return input;
    }

    /**
     * Sets input object for the tree viewer.
     * 
     * Must be called before dialog is opened.
     * 
     * @param inputElement
     */
    public void setInput(Object inputElement) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.input = inputElement;
    }

    public String getMessage() {
        return message;
    }

    /**
     * Sets optional message that is shown top of the tree.
     * 
     * Must be called before dialog is opened.
     * 
     * @param message
     */
    public void setMessage(String message) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.message = message;
    }

    public boolean isMultipleSelection() {
        return multipleSelection;
    }

    /**
     * Sets multi-selection support for the tree viewer. Single-selection is the default. 
     * 
     * Must be called before dialog is opened.
     * 
     * @param multipleSelection
     */
    public void setMultipleSelection(boolean multipleSelection) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.multipleSelection = multipleSelection;
    }
    
    /**
     * Use Double-click as way to close the dialog. This enabled by default.
     * 
     * Must be called before dialog is opened.
     * 
     * @param useDoubleClick
     */
    public void setUseDoubleClick(boolean useDoubleClick) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
		this.useDoubleClick = useDoubleClick;
	}
    
    public boolean isUseDoubleClick() {
		return useDoubleClick;
	}

    public boolean isExpandTree() {
        return expandTree;
    }

    /**
     * Set the tree viewer to fully expand.
     * 
     * Must be called before dialog is opened.
     * 
     * @param expandTree
     */
    public void setExpandTree(boolean expandTree) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.expandTree = expandTree;
    }
    
    /**
     * Sets sorter for the tree viewer.
     * 
     * Must be called before dialog is opened.
     * 
     * @param sorter
     */
    public void setSorter(ViewerSorter sorter) {
    	if (viewer != null)
    		throw new IllegalStateException("Dialog is already initialized");
        this.sorter =sorter;
    }
    
    public TreeViewer getViewer() {
		return viewer;
	}
    
}
