/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.strings.format;

import java.util.ArrayList;
import java.util.List;

/**
 * List of metrics formats
 * 
 * @author Toni Kalajainen
 */
public class MetricsFormatList {

    public static MetricsFormat METRICS_GENERIC = new MetricsFormat("%s", 1.0, "Generic");

    public static MetricsFormat METRICS_EXPONENT2 = new MetricsFormat("%e", 1.0, "Exponent");

    public static MetricsFormat METRICS_EXPONENT = new MetricsFormat("D0.##E0",1.0, "Exponent");

    public static MetricsFormat METRICS_DECIMAL = new MetricsFormat("D0.0###", 1.0, "Decimal");

    public static MetricsFormat METRICS_DECIMAL2 = new MetricsFormat("D0.####", 1.0, "Decimal");

    public static MetricsFormat METRICS_DECIMAL3 = new MetricsFormat("%,g", 1.0, "Decimal");

    public static MetricsFormat METRICS_DATE_FI = new MetricsFormat("%td.%tm.%ty", 1.0, "Date dd.mm.yyyy");

    public static MetricsFormat METRICS_DATE_IMPERIAL = new MetricsFormat("%tm/%td/%ty", 1.0, "Date mm/dd/yyyy");

    public static MetricsFormat METRICS_TIME_SECONDS = new MetricsFormat("%tS.%tL", 1.0, "Time, ss.ms");

    public static MetricsFormat METRICS_CLOCK_TIME = new MetricsFormat("%tH:%tM:%tS.%tL", 1.0, "Time, hh:mm:ss.ms");

    public static MetricsFormat METRICS_CLOCK_TIME2 = new MetricsFormat("%tH:%tM:%tS", 1.0, "Time, hh:mm:ss");

    // Templates
    private final List<MetricsFormat> formats = new ArrayList<MetricsFormat>();

    private final List<MetricsFormatListListener> listeners = new ArrayList<MetricsFormatListListener>();

    public MetricsFormat[] getFormats() {
        synchronized (formats) {
            return formats.toArray(new MetricsFormat[0]);
        }
    }

    public MetricsFormatList() {
    }

    private void addDefaults() {
        try {
            addFormat(METRICS_DECIMAL);
            addFormat(METRICS_DECIMAL2);
            addFormat(METRICS_DECIMAL3);
            addFormat(METRICS_EXPONENT);
            addFormat(METRICS_EXPONENT2);
            addFormat(METRICS_GENERIC);
            addFormat(METRICS_DATE_FI);
            addFormat(METRICS_DATE_IMPERIAL);
            addFormat(METRICS_TIME_SECONDS);
            addFormat(METRICS_CLOCK_TIME);
            addFormat(METRICS_CLOCK_TIME2);
        } catch (Exception e) {
        }
    }

    public void addListener(MetricsFormatListListener listener) {
        synchronized (this) {
            listeners.add(listener);
        }
    }

    public void removeListener(MetricsFormatListListener listener) {
        synchronized (this) {
            listeners.remove(listener);
        }
    }

    private void fireListChanged() {
        synchronized (this) {
            for (MetricsFormatListListener listener : listeners)
                listener.onListChanged(this);
        }
    }

    public MetricsFormat addFormat(MetricsFormat format) throws Exception {
        synchronized (this) {
            MetricsFormat old = findEqual(format);
            if (old != null)
                throw new Exception("Format already exists");
            formats.add(format);
            fireListChanged();
            return format;
        }
    }

    public void removeFormat(MetricsFormat format) throws Exception {
        synchronized (this) {
            int index = formats.indexOf(format);
            if (index < 0)
                throw new Exception("Format not found");
            formats.remove(format);
            fireListChanged();
        }
    }

    public MetricsFormat replaceFormatWith(MetricsFormat oldFormat, MetricsFormat newFormat) throws Exception {
        synchronized (this) {
            if (formats.contains(newFormat))
                throw new Exception("New format already exists");
            if (oldFormat == newFormat)
                throw new Exception("No changes");
            int index = formats.indexOf(oldFormat);
            if (index < 0)
                throw new Exception("Format not found");
            formats.remove(index);
            formats.add(index, newFormat);
            fireListChanged();
            return newFormat;
        }
    }

    public MetricsFormat findEqual(MetricsFormat format) {
        synchronized (this) {
            int i = formats.indexOf(format);
            if (i < 0)
                return null;
            return formats.get(i);
        }
    }

    public MetricsFormat findByIndex(int index) {
        return formats.get(index);
    }

    public int findIndex(MetricsFormat format) {
        for (int i=0; i<formats.size(); i++)
            if (formats.get(i).equals(format))
                return i;
        return -1;
    }

    private static MetricsFormatList list;

    public static MetricsFormatList getList() {
        return list;
    }

    static {
        list = new MetricsFormatList();
        list.addDefaults();
    }
}
