/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on Jan 21, 2005
 * 
 * Copyright Toni Kalajainen
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simantics.utils.strings;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.simantics.databoard.parser.StringEscapeUtils;
import org.simantics.utils.bytes.ByteArrays;


/**
 * Extended string functions
 * Byte array <-> string conversions
 *
 *  Length is not included
 *
 *
 * See also {@link StringEscapeUtils}
 * @author Toni Kalajainen
 */
public class EString {

    public static final char[] HEX_VALUES =
    {'0', '1', '2', '3' ,'4' ,'5', '6', '7', '8', '9' ,'A', 'B', 'C',
        'D', 'E', 'F'};


    /**
     * convert string to byte array
     * @param s
     * @return
     */
    public static byte[] toBytes(String s) {
        int size = s.length();
        byte array[] = new byte[size];
        // write chars
        for (int i=0; i<size; i++)
            array[i] = (byte) (s.charAt(i) );
        return array;
    }

    /**
     * convert bytearray to string
     * @param array
     * @return
     */
    public static String toString(byte array[]) {
        if (array==null) return null;
        int size = array.length;
        if (size==0) return null;
        // read chars
        //return new String(array, 4, size);
        char chars[] = new char[size];
        for (int i=0; i<size; i++)
            chars[i] = (char) (array[i] & 0xff);
        return new String(chars);
    }

    /**
     * read null terminated string
     * @param array
     * @param offset start offset
     * @return
     */
    public static String PChar(byte array[], int offset) {
        if (array==null) return null;
        int size = array.length;
        int zero = offset;
        while (zero<size && array[zero]!=0) zero++;
        if (zero>=size) throw new IndexOutOfBoundsException();
        char chars[] = new char[zero-offset];
        for (int i=0; i<chars.length; i++)
            chars[i] = (char) (array[i+offset] & 0xff);
        return new String(chars);
    }

    /**
     * read null terminated string
     * @param array
     * @param offset start offset
     * @param maxLen maximum bytes to read
     * @return
     */
    public static String PChar(byte array[], int offset, int maxLen) {
        if (array==null) return null;
        int size = array.length;
        if (offset+maxLen<size) size = offset+maxLen;
        int zero = offset;
        while (zero<size && array[zero]!=0) zero++;
        if (zero>=size) throw new IndexOutOfBoundsException();
        char chars[] = new char[zero-offset];
        for (int i=0; i<chars.length; i++)
            chars[i] = (char) (array[i+offset] & 0xff);
        return new String(chars);
    }

    /**
     * Explode words into array
     * @param str
     * @param breaker
     * @return
     */
    public static String[] explode(String str, String breaker) {
        return str.split(breaker);
    }
    /**
     * Explode string into lines
     * @param str
     * @return
     */
    public static String[] explode(String str) {
        return str.split("\n");
    }
    /**
     * implode array into string
     * @param strings
     * @param glue
     * @return
     */
    public static String implode(Object strings[], String glue) {
        if (strings.length==0) return null;
        if (strings.length == 1)
            return strings[0].toString();
        StringBuilder sb = new StringBuilder();
        sb.append(strings[0].toString());
        for (int i=1; i<strings.length; i++) {
            sb.append(glue);
            sb.append(strings[i]);
        }
        return sb.toString();
    }
    public static String implode(Collection<?> strings) {
        return implode(strings.toArray(new Object[strings.size()]));
    }
    public static String implode(Collection<?> strings, String glue) {
        return implode(strings.toArray(new Object[strings.size()]), glue);
    }
    /**
     * implode lines into array
     * @param strings
     * @return
     */
    public static String implode(Object strings[]) {
        if (strings.length==0) return "";

        String s0 = strings[0].toString();
        int s0len = s0.length();
        int len = s0len;
        for (int i=1; i<strings.length; i++)
            len += 1+strings[i].toString().length();

        char data[] = new char[len];
        s0.getChars(0, s0.length(), data, 0);
        int pos = s0len;

        for (int i=1; i<strings.length; i++) {
            data[pos] = '\n';
            String si = strings[i].toString();
            int silen = si.length();
            si.getChars(0, silen, data, pos+1);
            pos += 1+silen;
        }

        return new String(data);
    }

    /**
     * Add prefix to stings in array
     * @param strings
     * @param prefix
     * @return
     */
    public static String[] addPrefix(String strings[], String prefix) {
        String result[] = new String[strings.length];
        for (int i=0; i<result.length; i++)
            result[i] = prefix + strings[i];
        return result;
    }

    /**
     * add prefix to lines
     * @param strings
     * @param prefix
     * @return
     */
    public static String addPrefix(String strings, String prefix) {
        String result[] = explode(strings);
        for (int i=0; i<result.length; i++)
            result[i] = prefix + result[i];
        return implode(result);
    }

    /**
     * add prefix and suffix to lines
     * @param strings
     * @param prefix
     * @param suffixfix
     * @return
     */
    public static String addFix(String strings, String prefix, String suffix) {
        String result[] = explode(strings);
        for (int i=0; i<result.length; i++)
            result[i] = prefix + result[i] + suffix;
        return implode(result);
    }

    /**
     * Big Endian hex
     * @param value
     * @param decimals
     * @return
     */
    public static String intToBEHex(int value, int decimals) {
    	// TODO Replace with String.format("%0"+decimals+"X", value)
        String result="";
        for (int i=0; i<decimals; i++) {
            result += EString.HEX_VALUES[(value>>4) & 0xF];
            result += EString.HEX_VALUES[value & 0xF];
            value = value >> 8;
        }
        return result;
    }

    /**
     * splits line in a manner that is done to word wrap
     * Lines are broken between spaces if possible
     */
    public static String wordWrap(String text, int minWidth, int maxWidth)
    {
        // init values
        char cr = 0x0D;
        String lines[] = text.replaceAll(""+cr, "").split("\n");
        List<String> result = new ArrayList<String>();

        // iterate lines
        for (int i=0; i<lines.length; i++) {
            String line = lines[i];

            if (line.length() < maxWidth) {
                result.add(line);
                continue;
            }

            // Line is longer than maxwidth, split it
            String words[] = lines[i].split(" ");
            line = "";
            for (int j=0; j<words.length; j++) {
                // Nothing on the buffer
                if (line.equals("")) {
                    line = words[j];
                    while (line.length() > maxWidth) {
                        result.add(line.substring(0, maxWidth));
                        line = line.substring(maxWidth);
                    }
                    continue;
                }

                // Add to previous line buffer
                String word = words[j];

                // Check if adding this word fits maxwidth
                if (line.length()+1+word.length()<maxWidth) {
                    line = line + " " + word;
                    continue;
                }
                // Adding this word would make the line too long

                // Check if this line is long enough
                if (line.length()>=minWidth) {
                    result.add(line);
                    line = word;
                    continue;
                }
                // Line is not long enough with out AWord and with it
                // the line is too long.

                // So we need to split the line
                line = line + " " + word;
                while (line.length()>maxWidth) {
                    result.add(line.substring(0, maxWidth));
                    line = line.substring(maxWidth);
                }
            }
            if (!line.equals(""))
                result.add(line);
        }

        // Change string lines into single string
        StringBuilder sb = new StringBuilder();
        int rs = result.size();
        if (rs > 0) {
            sb.append(result.get(0));
            for (int i=1; i<rs; i++)
                sb.append("\n").append(result.get(i));
        }
        return sb.length() == 0 ? "" : sb.toString();
    }


    public static void printByteArray(byte array[]) {
        for (int i=0; i<array.length; i++) {
            System.out.print(array[i] & 0xff);
            if (i<array.length-1)
                System.out.print(",");
        }
    }

    public static void saveToFile(String text, String filename)
    throws IOException
    {
        ByteArrays.saveToFile(toBytes(text), filename);
    }

    /**
     * Escape characters in a string.
     * <p>
     * eg. value=5,5 will be escaped to value\=5\,5
     * with escape set =, and escape char \
     * 
     * @param str string to escape
     * @param escapeSet set of chars
     * @param escapeChar escape character
     * @return escaped string
     */
    public static String escapeString(String str, String escapeSet, char escapeChar)
    {
        escapeSet += escapeChar;
        StringBuilder sb = new StringBuilder(str.length()*2);
        for (int i=0; i<str.length(); i++)
        {
            char c = str.charAt(i);

            for (int j=0; j<escapeSet.length(); j++)
            {
                char ec = escapeSet.charAt(j);
                if (ec==c) {
                    sb.append(escapeChar);
                    break;
                }
            }

            sb.append(c);
        }

        return sb.toString();
    }

    /**
     * Unescape charaters in a string
     * <p>
     * eg. value\=5\,5 will be unescaped to value=5,5 with
     * escape set =, and escape char \
     * 
     * @param str string to unescape
     * @param escapeChar
     * @return unescaped string
     */
    public static String unescapeString(String str, char escapeChar)
    {
        StringBuilder sb = new StringBuilder(str.length());
        boolean prevWasEscapeChar = false;
        for (int i=0; i<str.length(); i++)
        {
            char c = str.charAt(i);

            // Atmost escape every second character
            if (prevWasEscapeChar || (c != escapeChar)) {
                prevWasEscapeChar = false;
                sb.append(c);
            } else {
                prevWasEscapeChar = true;
            }
        }

        return sb.toString();
    }

    /**
     * Scans escaped string
     * e.g. key=val\,ue,key2=xx returns key=val\,ue with endMark ,
     * 
     * @param str string to scan
     * @param escapeChar escape character
     * @param endMark end mark
     * @return everything before endMark
     */
    public static String scanEscapedString(String str, char escapeChar, char endMark)
    {
        StringBuilder sb = new StringBuilder(str.length());
        boolean prevWasEscapeChar = false;
        for (int i=0; i<str.length(); i++)
        {
            char c = str.charAt(i);

            // Atmost escape every second character
            if (prevWasEscapeChar || (c != escapeChar)) {
                if (!prevWasEscapeChar && c==endMark)
                    return sb.toString();
                // the next char won't be escape char
                prevWasEscapeChar = false;
            } else {
                // This is escape char
                prevWasEscapeChar = true;
            }
            sb.append(c);
        }

        return sb.toString();
    }

    /**
     * Compiles pattern from simple pattern. Simple pattern is normal
     * wild card compare that supports * and ? wild cards.
     * 
     * @param patternStr simple pattern
     * @return Regexp pattern
     */
    public static Pattern compileSimplePattern(String patternStr)
    throws PatternSyntaxException
    {
        String str ="";
        for (int i=0; i<patternStr.length(); i++)
        {
            char c = patternStr.charAt(i);
            if ( (c>='a'&&c<='z') || (c>='A'&&c<='Z') || (c>='0'&&c<='9'))
                str += c;
            else if ( c=='?' )
                str += ".?";
            else if ( c=='*' )
                str += ".*";
            else str += "\\"+c;
        }
        return Pattern.compile(str);
    }

    public static boolean simplePatternMatch(String str, String simplePattern)
    {
        try {
            Pattern ptr = compileSimplePattern(simplePattern);
            Matcher m = ptr.matcher(str);
            return m.matches();
        } catch (PatternSyntaxException pse) {
            return false;
        }
    }

    public static void main(String[] args) {
        System.out.println(escapeString("value=5,\\5", ",=", '\\'));
        System.out.println(unescapeString("value\\=5\\,\\\\5", '\\'));
        System.out.println(scanEscapedString("val\\,ue\\=5\\,\\\\5,value2=xxx", '\\', ','));
        System.out.println(scanEscapedString("\\,\\,,\\,\\,", '\\', ','));

        String value = "STRING 01234"+(char)(128)+(char)(129)+(char)(255);

        int X = 500;
        System.out.println(X+" = "+intToBEHex(X, 4));

        byte array[] = toBytes(value);
        System.out.print(value);
        System.out.print(" = ");
        printByteArray(array);
        System.out.println();

        @SuppressWarnings("unused")
        String str = toString(array);
        System.out.print(value);
        System.out.print(" = ");
        printByteArray(array);
        System.out.println();

        String text = "Reads b.length bytes from this file into the "+
        "byte array, starting at the current file pointer. This method"+
        " reads repeatedly from the file until the requested number of "+
        "bytes are read. This method blocks until the requested number of "+
        "bytes are read, the end of the stream is detected, or an exception "+
        "is thrown";
        text = wordWrap(text, 20, 30);
        System.out.println(text);


        List<String> v = new ArrayList<String>();
        v.add("jeps");
        v.add("jops");
        v.add("kops");
        v.add("hops");
        System.out.println(implode(v));
    }


}
