/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures.slice;

/**
 * @author Tuukka Lehtonen
 */
public class ValueRange {

    public static final ValueRange NO_RANGE = new ValueRange(0, Integer.MAX_VALUE);
    public static final ValueRange EMPTY = new ValueRange(0, 0);
    public static final ValueRange ONE   = new ValueRange(0, 1);
    public static final ValueRange TWO   = new ValueRange(0, 2);

    private final int start;
    private final int size;

    public static ValueRange make(int size) {
        if (size == 0)
            return EMPTY;
        else if (size == 1)
            return ONE;
        else if (size == 2)
            return TWO;
        return new ValueRange(size);
    }

    public static ValueRange make(int start, int size) {
        if (start == 0) {
            if (size == 0)
                return EMPTY;
            else if (size == 1)
                return ONE;
            else if (size == 2)
                return TWO;
        }
        return new ValueRange(start, size);
    }

    private ValueRange(int size) {
        this(0, size);
    }

    private ValueRange(int start, int size) {
        if (start < 0)
            throw new IllegalArgumentException("negative range start: " + start);
        if (size < 0)
            throw new IllegalArgumentException("negative range size: " + size);
        this.start = start;
        this.size = size;
    }

    public boolean isEmpty() {
        return size == 0;
    }

    public boolean isSingle() {
        return size == 1;
    }

    public int start() {
        return start;
    }

    public int end() {
        return start + size;
    }

    public int size() {
        return size;
    }

    @Override
    public String toString() {
        if (isEmpty())
            return "[]";
        if (isSingle())
            return "[" + start + "]";
        return "[" + start + "-" + (start + size - 1) + "]";
    }

    public String toString(int offset) {
        if (isEmpty())
            return "[]";
        if (isSingle())
            return "[" + (start+offset) + "]";
        return "[" + (start+offset) + "-" + (start + size - 1 + offset) + "]";
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + size;
        result = prime * result + start;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ValueRange other = (ValueRange) obj;
        return size == other.size && start == other.start;
    }

//    public ChildReference toChildReference() {
//        if (isEmpty())
//            throw new UnsupportedOperationException("cannot create a value reference for an empty range");
//        if (isSingle())
//            return new IndexReference(start);
//        //throw new RangeReference(start, start+size-1);
//        throw new UnsupportedOperationException("cannot create a value reference for a range with cardinality > 1");
//    }

    /**
     * @return a Simantics Databoard child reference string describing this range.
     */
    public String toChildReferenceString() {
        if (isEmpty())
            throw new UnsupportedOperationException("cannot create a value reference for an empty range");
        StringBuilder sb = new StringBuilder();
        if (isSingle())
            return sb.append("i-").append(start).toString();
        // TODO: proposal: http://dev.simantics.org/index.php/Talk:Databoard_Specification
        return sb.append("r-").append(start).append('-').append(start+size-1).toString();
    }

}
