/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.handler.e4;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.e4.core.contexts.Active;
import org.eclipse.e4.core.di.annotations.CanExecute;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.db.ChangeSetIdentifier;
import org.simantics.db.Operation;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.NoHistoryException;
import org.simantics.db.service.ManagementSupport;
import org.simantics.db.service.UndoRedoSupport;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * @author Kalle Kondelin
 */
public class SessionRedoHandler {

    private final boolean DEBUG = false;

    @CanExecute
    public boolean canExecute() {
        return UndoRedoTester.canRedo();
    }
    
    @Execute
    public void execute(@Active MPart activePart) throws ExecutionException {
        if (DEBUG)
            System.out.println("--\nRedo handler called.");

        if (activePart == null)
            return;
        final Session session = Simantics.peekSession();
        if (session == null)
            return;

        try {
            final AtomicReference<String> msg = new AtomicReference<String>();
            IRunnableWithProgress redo = new IRunnableWithProgress() {
                @Override
                public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                    try {
                        monitor.beginTask("Redo", IProgressMonitor.UNKNOWN);
                        msg.set( redo( session ) );
                    } catch (NoHistoryException e) {
                        msg.set("Nothing to redo.");
                    } catch (DatabaseException e) {
                        throw new InvocationTargetException(e);
                    } finally {
                        monitor.done();
                    }
                }
            };

            // busyCursorWhile does not work because locking the session for undo
            // will also lock the UI completely.
            //PlatformUI.getWorkbench().getProgressService().busyCursorWhile(undo);
            Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
            new ProgressMonitorDialog(shell).run(true, false, redo);
            
            //
            // TODO Not the Eclipse 4 way of using IStatusLineManager!
            // See:
            // https://bugs.eclipse.org/bugs/show_bug.cgi?id=332499
            // https://www.eclipse.org/forums/index.php?t=msg&th=367379&goto=941232&#msg_941232
            //
            IWorkbenchPart part = WorkbenchUtils.getActiveWorkbenchPart();
            IStatusLineManager manager = WorkbenchUtils.getStatusLine(part);
            manager.setMessage(msg.get());
//            actionBars.getStatusLineManager().setMessage( msg.get() );
        } catch (InvocationTargetException e1) {
            throw new ExecutionException("Redo failed, database failure.", e1.getTargetException());
        } catch (InterruptedException e1) {
            throw new ExecutionException("Redo failed, interrupted.", e1);
        }
        return;
    }

    private String redo(Session session) throws DatabaseException {
        UndoRedoSupport support = session.getService(UndoRedoSupport.class);
        List<Operation> ops = support.redo(session, 1);
        if(ops.isEmpty())
            return "Nothing to redo.";
        
        Operation mainOperation = ops.get(0);
        
        String msg = null;
        long csId = mainOperation.getCSId();

        ManagementSupport management = session.getService(ManagementSupport.class);
        for(ChangeSetIdentifier id : management.getChangeSetIdentifiers(csId, csId))
            if(msg == null)
                msg = "Redid: " + SessionUndoHandler.getComment(session, id);

        if (DEBUG)
            System.out.println(msg);
        return msg;
    }

}
