/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import java.util.List;

import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveListener4;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.contexts.IContextActivation;
import org.eclipse.ui.contexts.IContextService;
import org.simantics.utils.datastructures.MapList;

/**
 * @author Tuukka Lehtonen
 */
public class PerspectiveContextActivator implements IPerspectiveListener4, IWindowListener {

    private IWorkbenchWindow                    activeWindow;

    private String                              oldPerspective;

    private MapList<String, IContextActivation> activations = new MapList<String, IContextActivation>();


    public PerspectiveContextActivator() {
        PlatformUI.getWorkbench().addWindowListener(this);
    }

    public void dispose() {
        PlatformUI.getWorkbench().removeWindowListener(this);
    }

    //------------------------------------------------------------------------
    // IPerspectiveListener4
    //------------------------------------------------------------------------

    @Override
    public void perspectivePreDeactivate(IWorkbenchPage page, IPerspectiveDescriptor perspective) {
    }

    @Override
    public void perspectiveClosed(IWorkbenchPage page, IPerspectiveDescriptor perspective) {
    }

    @Override
    public void perspectiveDeactivated(IWorkbenchPage page, IPerspectiveDescriptor perspective) {
    }

    @Override
    public void perspectiveOpened(IWorkbenchPage page, IPerspectiveDescriptor perspective) {
    }

    @Override
    public void perspectiveSavedAs(IWorkbenchPage page, IPerspectiveDescriptor oldPerspective,
            IPerspectiveDescriptor newPerspective) {
    }

    @Override
    public void perspectiveChanged(IWorkbenchPage page, IPerspectiveDescriptor perspective,
            IWorkbenchPartReference partRef, String changeId) {
        // See IWorkbenchPage.CHANGED_* constants for change id's.
    }

    @Override
    public void perspectiveActivated(IWorkbenchPage page, IPerspectiveDescriptor perspective) {
        activatePerspective(perspective.getId());
    }

    @Override
    public void perspectiveChanged(IWorkbenchPage page, IPerspectiveDescriptor perspective, String changeId) {
        // See IWorkbenchPage.CHANGED_* constants for change id's.
    }

    //------------------------------------------------------------------------
    // IWindowListener
    //------------------------------------------------------------------------

    @Override
    public void windowActivated(IWorkbenchWindow window) {
//        System.out.println("attaching to window: " + window);
        attachToWindow(window);
    }

    @Override
    public void windowClosed(IWorkbenchWindow window) {
    }

    @Override
    public void windowDeactivated(IWorkbenchWindow window) {
//        System.out.println("detaching from window: " + window);
        detachFromWindow(window);
    }

    @Override
    public void windowOpened(IWorkbenchWindow window) {
    }

    //------------------------------------------------------------------------
    // UTILITIES
    //------------------------------------------------------------------------

    private void attachToWindow(IWorkbenchWindow window) {
        activeWindow = window;
        window.addPerspectiveListener(this);
        IPerspectiveDescriptor perspective = window.getActivePage().getPerspective();
        if (perspective != null) {
            activatePerspective(perspective.getId());
        }
    }

    private void detachFromWindow(IWorkbenchWindow window) {
        window.removePerspectiveListener(this);
    }

    private IContextService getService(IWorkbenchWindow window) {
        return (IContextService) window.getWorkbench().getService(IContextService.class);
        //return (IContextService) window.getService(IContextService.class);
    }

    private IContextActivation activate(String ctx) {
        IContextService contextService = getService(activeWindow);
        if (contextService != null && ctx != null) {
            IContextActivation act = contextService.activateContext(ctx);
//            System.out.println("activating context: " + act);
            return act;
        }
        return null;
    }

    private void deactivate(IContextActivation activation) {
        IContextService contextService = getService(activeWindow);
        if (contextService != null && activation != null) {
//            System.out.println("deactivating context: " + activation);
            contextService.deactivateContext(activation);
        }
    }

    private void activatePerspective(String perspectiveId) {
//        System.out.println("activating perspective: " + perspectiveId + " (old=" + oldPerspective + ")");

        if (oldPerspective != null) {
            if (oldPerspective.equals(perspectiveId))
                return;

            List<IContextActivation> acts = activations.getValues(oldPerspective);
            if (acts != null) {
                activations.remove(oldPerspective);
                for (IContextActivation act : acts) {
                    deactivate(act);
                }
            }
        }
        List<IPerspectiveContextExtension> exts = PerspectiveContextBindingManager.getInstance().getExtensions(perspectiveId);
        for (IPerspectiveContextExtension ext : exts) {
            for (String ctx : ext.getContextIds()) {
                IContextActivation activation = activate(ctx);
                if (activation != null) {
                    activations.add(perspectiveId, activation);
                }
            }
        }
        oldPerspective = perspectiveId;
    }

}
