/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.trend.configuration;

import org.simantics.databoard.annotations.Identifier;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.util.Bean;
import org.simantics.utils.strings.AlphanumComparator;

/**
 * @author Toni Kalajainen
 */
public class TrendItem extends Bean {
	
	// Sorting and color index
	public int index;
	
	// Chart Label
	public @Optional String label;	
	// Normal Label (eg. The label in CSV file)
	public @Optional String simpleLabel;
	// Variable reference in user friendly format
	public @Optional String variableReference;
	
	// GroupId in the history
	public @Identifier String groupId;
	// ItemId in the group
	public @Identifier String groupItemId;
	// Variable id in format understood by org.simantics.simulation.data.Datasource
	public @Identifier String variableId;
	
	public Scale scale;
	
	// TODO Rename Binary to Bar
	public enum Renderer { Analog, Binary } public Renderer renderer;
	
	public @Optional String unit;
	
	public DrawMode drawMode = DrawMode.getDefault();

	// If true, the item shall not be shown in the chart.
	public boolean hidden = false;

	// Defines a custom stroke to be used when rendering this item.
	public @Optional Float customStrokeWidth;

	// Defines a custom color to be used when rendering this item.
	// If not defined, color is calculated based on #index.
	public @Optional float[] customColor;

	public enum DrawMode {
		DeviationAndLine,
		DeviationAndAverage,
		DeviationAndSample,
		Average,
		Line,
		Sample,
		Deviation;
		public static DrawMode getDefault() {
			return DeviationAndLine; 
		}
	}
	
	public TrendItem() {
	}

	public TrendItem(int index, String label, String subscriptionId, String variableId, Scale scale) {
		this(index, label, subscriptionId, variableId, scale, Renderer.Analog);
	}

	public TrendItem(int index, String label, String subscriptionId, String variableId, Scale scale, Renderer renderer) {
		this.index = index;
		this.label = label;
		this.variableId = variableId;
		this.scale = scale;
		this.renderer = renderer;
		this.groupId = subscriptionId;
		this.groupItemId = variableId;
	}	

	public TrendItem(int index, String label, String subscriptionId, String variableId, Scale scale, Renderer renderer, double min, double max) {
		this.index = index;
		this.label = label;
		this.variableId = variableId;
		this.scale = new Scale.Manual(min, max);
		this.renderer = renderer;
		this.groupId = subscriptionId;
		this.groupItemId = variableId;
	}
	
	public TrendItem(int index, String label, String subscriptionId, String variableId, Scale scale, Renderer renderer, double min, double max, DrawMode drawMode) {
		this.index = index;
		this.label = label;
		this.variableId = variableId;
		this.scale = new Scale.Manual(min, max);
		this.renderer = renderer;
		this.groupId = subscriptionId;
		this.groupItemId = variableId;
		this.drawMode = drawMode;
	}
	
	@Override
	public int compareTo(Bean o) {
		if ( o instanceof TrendItem == false ) return 0;
		TrendItem other = (TrendItem) o;
		
		int c = renderer.ordinal() - other.renderer.ordinal();
		if ( c!=0 ) return c;
		
		c = index - other.index;
		if ( c!=0 ) return c;
		
		String myLabel1 = label!=null?label:"";
		String otLabel1 = other.label!=null?other.label:"";
		c = AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(myLabel1, otLabel1);
		if ( c!=0 ) return c;
		
		String myLabel2 = simpleLabel!=null?simpleLabel:"";
		String otLabel2 = other.simpleLabel!=null?other.simpleLabel:"";
		c = AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(myLabel2, otLabel2);
		if ( c!=0 ) return c;
		
		String mySid = groupId!=null?groupId:""; 
		String otSid = other.groupId!=null?other.groupId:""; 
		c = mySid.compareTo(otSid);
		if ( c!=0 ) return c;

		String myGiid = groupItemId!=null?groupItemId:""; 
		String otGiid = other.groupItemId!=null?other.groupItemId:""; 
		c = myGiid.compareTo(otGiid);
		if ( c!=0 ) return c;

		String myVid = variableId!=null?variableId:""; 
		String otVid = other.variableId!=null?other.variableId:""; 
		c = myVid.compareTo(otVid);
		if ( c!=0 ) return c;
		
		return c;		
	}
	
}

