package org.simantics.tests.modelled.ui;

import java.io.IOException;

import org.eclipse.ui.IPersistableElement;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.options.ModuleCompilationOptions;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.source.ModuleSource;
import org.simantics.scl.compiler.source.TextualModuleSource;
import org.simantics.scl.ui.editor2.StandardSCLModuleEditorInput;
import org.simantics.tests.modelled.ontology.TestsResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class STSTestEditorInput extends StandardSCLModuleEditorInput  {

    private static final Logger LOGGER = LoggerFactory.getLogger(STSTestEditorInput.class);
    
    public STSTestEditorInput(String moduleName) {
        super(moduleName);
    }
    
    @Override
    public boolean exists() {
        return true;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public <T> T getAdapter(Class<T> adapter) {
        if(adapter.equals(ModuleSource.class))
            try {
                return (T) Simantics.getSession().syncRequest(new ReadSTSModuleSource(getModuleName()));
            } catch (DatabaseException e) {
                e.printStackTrace();
            }
        if(adapter.equals(IPersistableElement.class))
            return (T)this;
        return null;
    }
    
    
    private static class STSTextualModuleSource extends TextualModuleSource {

        private String moduleText;

        public STSTextualModuleSource(String stsModuleURI, String moduleText) {
            super(stsModuleURI);
            this.moduleText = moduleText;
        }

        @Override
        public void update(String newSourceText) {
            try {
                Simantics.getSession().syncRequest(new WriteSTSModuleSource(getModuleName(), newSourceText));
            } catch (DatabaseException e) {
                LOGGER.error("", e);
            }
        }
        
        @Override
        public Failable<Module> compileModule(ModuleRepository environment, UpdateListener listener, ModuleCompilationOptions options) {
            return super.compileModule(environment, listener, options);
        }
        
        @Override
        public boolean isUpdateable() {
            return true;
        }

        @Override
        public String getSourceText(UpdateListener listener) throws IOException {
            return moduleText;
        }
    }
    
    static class ReadSTSModuleSource extends UnaryRead<String, ModuleSource> {
        
        public ReadSTSModuleSource(String moduleName) {
            super(moduleName);
        }

        @Override
        public ModuleSource perform(ReadGraph graph) throws DatabaseException {
            Resource moduleResource = graph.getPossibleResource(parameter);
            if(moduleResource == null)
                return null;
            TestsResource TESTS = TestsResource.getInstance(graph);
            if(!graph.isInstanceOf(moduleResource, TESTS.STSTest))
                return null;
            String text = graph.getRelatedValue(moduleResource, TESTS.STSTest_definition);
            return new STSTextualModuleSource(parameter, text);
        }
    }
    
    static class WriteSTSModuleSource extends WriteRequest {
        private final String moduleURI;
        private final String sourceText;
        
        public WriteSTSModuleSource(String moduleURI, String sourceText) {
            this.moduleURI = moduleURI;
            this.sourceText = sourceText;
        }

        @Override
        public void perform(WriteGraph graph) throws DatabaseException {
            Resource moduleResource = graph.getPossibleResource(moduleURI);
            if(moduleResource == null)
                return;
            TestsResource TESTS = TestsResource.getInstance(graph);
            if(!graph.isInstanceOf(moduleResource, TESTS.STSTest))
                return;
            graph.claimLiteral(moduleResource, TESTS.STSTest_definition, sourceText);
        }
    }
    
    @Override
    public String getFactoryId() {
        return "org.simantics.tests.modelled.ui.stseditor.inputFactory";
    }

}
