/*******************************************************************************
 * Copyright (c) 2000, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.simantics.tests.modelled.ui;


import java.text.MessageFormat;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * A panel with counters for the number of Runs, Errors and Failures.
 */
public class STSCounterPanel extends Composite {
    protected Text numberOfErrors;
    protected Text numberOfSkipped;
    protected Text numberOfFailures;
    protected Text numberOfRuns;
    protected int total;
    protected int ignoredCount;
    protected int assumptionFailedCount;

    private final Image fErrorIcon = null;
    private final Image fFailureIcon = null;

    public STSCounterPanel(Composite parent) {
        super(parent, SWT.WRAP);
        GridLayout gridLayout= new GridLayout();
        gridLayout.numColumns= 9;
        gridLayout.makeColumnsEqualWidth= false;
        gridLayout.marginWidth= 0;
        setLayout(gridLayout);

        numberOfRuns= createLabel("Runs:", null, " 0/0  "); //$NON-NLS-1$
        numberOfSkipped = createLabel("Skipped:", null," 0 ");
        numberOfErrors= createLabel("Errors:", fErrorIcon, " 0 "); //$NON-NLS-1$
        numberOfFailures= createLabel("Failures:", fFailureIcon, " 0 "); //$NON-NLS-1$

        addDisposeListener(new DisposeListener() {
            public void widgetDisposed(DisposeEvent e) {
                disposeIcons();
            }
        });
    }

    private void disposeIcons() {
        if (fErrorIcon != null)
            fErrorIcon.dispose();
        if (fFailureIcon != null)
            fFailureIcon.dispose();
    }

    private Text createLabel(String name, Image image, String init) {
        Label label= new Label(this, SWT.NONE);
        if (image != null) {
            image.setBackground(label.getBackground());
            label.setImage(image);
        }
        label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));

        label = new Label(this, SWT.NONE);
        label.setText(name);
        label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
        //label.setFont(JFaceResources.getBannerFont());

        Text value = new Text(this, SWT.READ_ONLY);
        value.setText(init);
        // bug: 39661 Junit test counters do not repaint correctly [JUnit]
//        SWTUtil.fixReadonlyTextBackground(value);
        value.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.HORIZONTAL_ALIGN_BEGINNING));
        return value;
    }

    public void reset() {
        setErrorValue(0);
        setFailureValue(0);
        setRunValue(0, 0, 0);
        total= 0;
    }

    public void setTotal(int value) {
        total= value;
    }

    public int getTotal(){
        return total;
    }

    public void setRunValue(int value, int ignoredCount, int assumptionFailureCount) {
        String runString;
        String runStringTooltip;
        if (ignoredCount == 0 && assumptionFailureCount == 0) {
            runString= Messages.format("{0}/{1}", new String[] { Integer.toString(value), Integer.toString(total) });
            runStringTooltip= runString;
        } else if (ignoredCount != 0 && assumptionFailureCount == 0) {
            runString= Messages.format("Skipped", new String[] { Integer.toString(value), Integer.toString(total), Integer.toString(ignoredCount) });
            runStringTooltip= Messages.format("Ignored", new String[] { Integer.toString(value), Integer.toString(total), Integer.toString(ignoredCount) });
        } else if (ignoredCount == 0 && assumptionFailureCount != 0) {
            runString= Messages.format("Skipped", new String[] { Integer.toString(value), Integer.toString(total), Integer.toString(assumptionFailureCount) });
            runStringTooltip= Messages.format("Failed", new String[] { Integer.toString(value), Integer.toString(total), Integer.toString(assumptionFailureCount) });
        } else {
            runString= Messages.format("Skipped", new String[] { Integer.toString(value), Integer.toString(total), Integer.toString(ignoredCount + assumptionFailureCount) });
            runStringTooltip= Messages.format("Failed", new String[] { Integer.toString(value), Integer.toString(total), Integer.toString(ignoredCount), Integer.toString(assumptionFailureCount) });
        }
        numberOfRuns.setText(runString);
        numberOfRuns.setToolTipText(runStringTooltip);

        if (ignoredCount == 0 && ignoredCount > 0  || ignoredCount != 0 && ignoredCount == 0) {
            layout();
        } else if (assumptionFailedCount == 0 && assumptionFailureCount > 0 || assumptionFailedCount != 0 && assumptionFailureCount == 0) {
            layout();
        } else {
            numberOfRuns.redraw();
            redraw();
        }
        assumptionFailedCount= assumptionFailureCount;
    }

    public void setIgnoredValue(int value) {
        numberOfSkipped.setText(Integer.toString(value));
        redraw();
    }
    
    public void setErrorValue(int value) {
        numberOfErrors.setText(Integer.toString(value));
        redraw();
    }

    public void setFailureValue(int value) {
        numberOfFailures.setText(Integer.toString(value));
        redraw();
    }
    
    private static class Messages {

        public static String format(String message, Object object) {
            return MessageFormat.format(message, new Object[] { object});
        }

        public static String format(String message, Object[] objects) {
            return MessageFormat.format(message, objects);
        }

        private Messages() {
            // Not for instantiation
        }
    }
}
