package org.simantics.tests.modelled.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.scl.compiler.commands.CommandSession;
import org.simantics.scl.compiler.commands.TestScriptExecutor;
import org.simantics.scl.compiler.elaboration.java.Builtins;
import org.simantics.scl.compiler.elaboration.java.JavaModule;
import org.simantics.scl.compiler.module.coverage.CombinedCoverage;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;

public class ModelledSTSTest {

    private final String name;
    private final String parentName;
    private final String code;
    private final int priority;
    private final boolean ignored;
    private Set<String> imports;
    private Set<String> dependencies;
    private Set<String> unresolvedDependencies;

    private CombinedCoverage coverage;
    private Map<String, String> variables;

    ModelledSTSTest(String name, String parentName, String code, int priority, boolean ignored, Set<String> imports, Set<String> dependencies, Map<String, String> variables) {
        this.name = name;
        this.parentName = parentName;
        this.code = code;
        this.priority = priority;
        this.ignored = ignored;
        this.imports = imports;
        this.dependencies = dependencies;
        this.unresolvedDependencies = new HashSet<>(dependencies);
        this.variables = variables;
    }

    public String getName() {
        return name;
    }

    public String getParentName() {
        return parentName;
    }

    public String getCode() {
        return code;
    }

    public int getPriority() {
        return priority;
    }

    public boolean isIgnored() {
        return (ignored || !unresolvedDependencies.isEmpty());
    }

    public void setCoverage(CombinedCoverage coverage) {
        this.coverage = coverage;
    }

    public CombinedCoverage getCoverage() {
        return coverage;
    }

    public static class CommandSessionVariable {

        private final String name;
        private final Type type;
        private final Object value;

        public CommandSessionVariable(String name, Type type, Object value) {
            this.name = name;
            this.type = type;
            this.value = value;
        }

        public String getName() {
            return name;
        }

        public Type getType() {
            return type;
        }

        public Object getValue() {
            return value;
        }
    }

    public List<CommandSessionVariable> run(List<CommandSessionVariable> vars) throws IOException {
//        ModuleRepository repo = new ModuleRepository(SCLOsgi.SOURCE_REPOSITORY);
    	ModuleRepository repo = SCLOsgi.MODULE_REPOSITORY;
        CommandSession session = null;
        try {
//            repo.setAdvisor(new ModuleCompilationOptionsAdvisor() {
//
//                @Override
//                public ModuleCompilationOptions getOptions(String moduleName) {
//                    // TODO: default to false
//                    boolean coverage = true;
//                    // TODO: add moduleName filters back
//    //                for (Pattern p : getModuleNameFilterPatterns()) {
//    //                    if (p.matcher(moduleName.toLowerCase()).find()) {
//    //                        coverage = true;
//    //                        break;
//    //                    }
//    //                }
//                    return new ModuleCompilationOptions(coverage);
    //                    }
//            });

            SCLReportingHandler handler = (SCLReportingHandler) SCLContext.getCurrent().get(SCLReportingHandler.REPORTING_HANDLER);
            session = new CommandSession(repo, handler);

            for(String imp : imports) {
                session.execute("import \"" + imp + "\"");
            }

            for (CommandSessionVariable var : vars)
                session.setVariable(var.getName(), var.getType(), var.getValue());

            for (Map.Entry<String, String> entry : variables.entrySet())
                session.setVariable(entry.getKey(), Types.STRING, entry.getValue());

            new TestScriptExecutor(session, new BufferedReader(new StringReader(code)), handler, false).execute();
            STSSuiteTestCollector.setTestCoverage(this, session);

            // Return variables from this session
            List<CommandSessionVariable> result = new ArrayList<>();
            for (String var : session.getVariables())
                result.add(new CommandSessionVariable(var, session.getVariableType(var), session.getVariableValue(var)));

            return result;
        } finally {
            // remember to flush this repository
//            repo.flush();
            Builtins.flush();
            JavaModule.flush();
        }
    }

    public Set<String> getDependencies() {
        return dependencies;
    }

    public Set<String> getUnresolvedDependencies() {
        return unresolvedDependencies;
    }

    public boolean resolveDependency(String testDep) {
        return unresolvedDependencies.remove(testDep);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((code == null) ? 0 : code.hashCode());
        result = prime * result + ((coverage == null) ? 0 : coverage.hashCode());
        result = prime * result + ((imports == null) ? 0 : imports.hashCode());
        result = prime * result + ((dependencies == null) ? 0 : dependencies.hashCode());
        result = prime * result + (ignored ? 1231 : 1237);
        result = prime * result + ((name == null) ? 0 : name.hashCode());
        result = prime * result + priority;
        result = prime * result + ((unresolvedDependencies == null) ? 0 : unresolvedDependencies.hashCode());
        result = prime * result + ((variables == null) ? 0 : variables.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ModelledSTSTest other = (ModelledSTSTest) obj;
        if (code == null) {
            if (other.code != null)
                return false;
        } else if (!code.equals(other.code))
            return false;
        if (coverage == null) {
            if (other.coverage != null)
                return false;
        } else if (!coverage.equals(other.coverage))
            return false;
        if (imports== null) {
            if (other.imports != null)
                return false;
        } else if (!imports.equals(other.imports))
            return false;
        if (dependencies == null) {
            if (other.dependencies != null)
                return false;
        } else if (!dependencies.equals(other.dependencies))
            return false;
        if (ignored != other.ignored)
            return false;
        if (name == null) {
            if (other.name != null)
                return false;
        } else if (!name.equals(other.name))
            return false;
        if (priority != other.priority)
            return false;
        if (unresolvedDependencies == null) {
            if (other.unresolvedDependencies != null)
                return false;
        } else if (!unresolvedDependencies.equals(other.unresolvedDependencies))
            return false;
        if (variables == null) {
            if (other.variables != null)
                return false;
        } else if (!variables.equals(other.variables))
            return false;
        return true;
    }
}
