package org.simantics.scl.ui.console;

import org.eclipse.jface.preference.BooleanFieldEditor;
import org.eclipse.jface.preference.FieldEditor;
import org.eclipse.jface.preference.FieldEditorPreferencePage;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.IntegerFieldEditor;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.simantics.scl.ui.Activator;

/**
 * A page to set the preferences for the SCL console.
 * 
 * @author Tuukka Lehtonen
 * @since 1.42.0
 * @see SCLConsolePreferenceConstants 
 */
public class SCLConsolePreferencePage extends FieldEditorPreferencePage implements IWorkbenchPreferencePage {

    /**
     * A boolean field editor that provides access to this editors boolean button.
     * 
     * Copied from
     * <code>org.eclipse.debug.internal.ui.preferences.BooleanFieldEditor2</code>.
     */
    public class BooleanFieldEditor2 extends BooleanFieldEditor {
        private  Button fChangeControl;

        public BooleanFieldEditor2(
                String name,
                String labelText,
                int style,
                Composite parent) {
            super(name, labelText, style, parent);
        }

        @Override
        public Button getChangeControl(Composite parent) {
            if (fChangeControl == null) {
                fChangeControl = super.getChangeControl(parent);
            }
            return fChangeControl;
        }
    }

    /**
     * This class exists to provide visibility to the <code>refreshValidState</code>
     * method and to perform more intelligent clearing of the error message.
     */
    protected class ConsoleIntegerFieldEditor extends IntegerFieldEditor {

        public ConsoleIntegerFieldEditor(String name, String labelText, Composite parent) {
            super(name, labelText, parent);
        }

        /**
         * @see org.eclipse.jface.preference.FieldEditor#refreshValidState()
         */
        @Override
        protected void refreshValidState() {
            super.refreshValidState();
        }

        /**
         * Clears the error message from the message line if the error
         * message is the error message from this field editor.
         */
        @Override
        protected void clearErrorMessage() {
            if (canClearErrorMessage()) {
                super.clearErrorMessage();
            }
        }
    }

    private BooleanFieldEditor2 fUseBufferSize = null;
    private ConsoleIntegerFieldEditor fBufferSizeEditor = null;

    /**
     * Create the console page.
     */
    public SCLConsolePreferencePage() {
        super(GRID);
        setDescription(Messages.SCLConsolePreferencePage_Description);
        setPreferenceStore(Activator.getInstance().getPreferenceStore());
    }

    /**
     * Create all field editors for this page
     */
    @Override
    public void createFieldEditors() {
        fUseBufferSize = new BooleanFieldEditor2(Preferences.CONSOLE_LIMIT_CONSOLE_OUTPUT, Messages.SCLConsolePreferencePage_Limit_console_output_Label, SWT.NONE, getFieldEditorParent());
        addField(fUseBufferSize);

        fBufferSizeEditor = new ConsoleIntegerFieldEditor(Preferences.CONSOLE_LOW_WATER_MARK, Messages.SCLConsolePreferencePage_Console_buffer_size_Label, getFieldEditorParent());
        addField(fBufferSizeEditor);
        fBufferSizeEditor.setValidRange(1000, 10000000);
        fBufferSizeEditor.setErrorMessage(Messages.SCLConsolePreferencePage_Console_buffer_size_Error);

        fUseBufferSize.getChangeControl(getFieldEditorParent()).addSelectionListener(
                SelectionListener.widgetSelectedAdapter(e -> updateBufferSizeEditor()));
    }

    /**
     * @see IWorkbenchPreferencePage#init(IWorkbench)
     */
    @Override
    public void init(IWorkbench workbench) {
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.preference.IPreferencePage#performOk()
     */
    @Override
    public boolean performOk() {
        boolean ok = super.performOk();

        // update high water mark to be (about) 100 lines (100 * 80 chars) greater than low water mark
        IPreferenceStore store = Activator.getInstance().getPreferenceStore();
        int low = store.getInt(Preferences.CONSOLE_LOW_WATER_MARK);
        store.setValue(Preferences.CONSOLE_HIGH_WATER_MARK, Preferences.highWatermarkForLow(low));

        return ok;
    }

    /**
     * @see org.eclipse.jface.preference.FieldEditorPreferencePage#initialize()
     */
    @Override
    protected void initialize() {
        super.initialize();
        updateBufferSizeEditor();
    }

    /**
     * Update enablement of buffer size editor based on enablement of 'limit
     * console output' editor.
     */
    protected void updateBufferSizeEditor() {
        Button b = fUseBufferSize.getChangeControl(getFieldEditorParent());
        fBufferSizeEditor.getTextControl(getFieldEditorParent()).setEnabled(b.getSelection());
        fBufferSizeEditor.getLabelControl(getFieldEditorParent()).setEnabled(b.getSelection());
    }

    /**
     * @see org.eclipse.jface.preference.PreferencePage#performDefaults()
     */
    @Override
    protected void performDefaults() {
        super.performDefaults();
        updateBufferSizeEditor();
    }

    protected boolean canClearErrorMessage() {
        return fBufferSizeEditor.isValid();
    }

    /**
     * @see org.eclipse.jface.util.IPropertyChangeListener#propertyChange(org.eclipse.jface.util.PropertyChangeEvent)
     */
    @Override
    public void propertyChange(PropertyChangeEvent event) {
        if (event.getProperty().equals(FieldEditor.IS_VALID)) {
            boolean newValue = ((Boolean) event.getNewValue()).booleanValue();
            // If the new value is true then we must check all field editors.
            // If it is false, then the page is invalid in any case.
            if (newValue) {
                if (fBufferSizeEditor != null && event.getSource() != fBufferSizeEditor) {
                    fBufferSizeEditor.refreshValidState();
                }
                checkState();
            } else {
                super.propertyChange(event);
            }

        } else {
            super.propertyChange(event);
        }
    }

}
