package org.simantics.scl.compiler.internal.parsing.documentation;

import java.io.IOException;
import java.io.StringReader;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.simantics.scl.compiler.elaboration.modules.Documentation;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.util.TypeUnparsingContext;

public abstract class HtmlUnparsingContext {
    StringBuilder stringBuilder;
    String listLevel = "";
    
    int tableOfContentsLevel = 0;
    StringBuilder tableOfContents;
    
    int headerLinkId = 0;
    
    public HtmlUnparsingContext(StringBuilder stringBuilder) {
        this.stringBuilder = stringBuilder;
        this.tableOfContents = null;
    }
    
    public HtmlUnparsingContext() {
        this(new StringBuilder());
        tableOfContents = new StringBuilder();
    }

    public StringBuilder getStringBuilder() {
        return stringBuilder;
    }
    
    public StringBuilder getTableOfContents() {
        return tableOfContents;
    }
    
    public void finishTableOfContents() {
        while(tableOfContentsLevel > 0) {
            tableOfContents.append("</ul>\n");
            --tableOfContentsLevel;
        }
    }
    
    public void header(int level, String text) {
        clear();        
        ++headerLinkId;
        stringBuilder.append("<h").append(level)
        .append(" id=\"sec").append(headerLinkId).append("\">")
        .append(text)
        .append("</h").append(level).append(">\n");
        
        while(tableOfContentsLevel > level) {
            tableOfContents.append("</ul>\n");
            --tableOfContentsLevel;
        }
        while(tableOfContentsLevel < level) {
            tableOfContents.append("<ul>\n");
            ++tableOfContentsLevel;
        }
        tableOfContents.append("<li><a href=\"#sec")
            .append(headerLinkId).append("\">")
            .append(text).append("</a></li>\n");
    }
        
    public void clear() {
        setListLevel("");
    }
    
    public void setListLevel(String newLevel) {
        String oldLevel = this.listLevel;
        int commonLevel = 0;
        while(commonLevel < oldLevel.length() && commonLevel < newLevel.length() && 
                newLevel.charAt(commonLevel) == oldLevel.charAt(commonLevel))
            ++commonLevel;
        
        for(int i=oldLevel.length()-1;i>=commonLevel;--i) {
            char c = oldLevel.charAt(i);
            if(c == '*')
                stringBuilder.append("</ul>\n");
            else if(c == '#')
                stringBuilder.append("</ol>\n");
        }
        for(int i=commonLevel;i<newLevel.length();++i) {
            char c = newLevel.charAt(i);
            if(c == '*')
                stringBuilder.append("<ul>\n");
            else if(c == '#')
                stringBuilder.append("<ol>\n");
        }
        
        this.listLevel = newLevel;
    }
    
    public void appendDocumentation(String doc) {
        DocumentationLexer lexer = new DocumentationLexer(new StringReader(doc));

        clear();
        while(true) {
            DocumentationElement element = null;
            try {
                element = lexer.nextToken();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (DocumentationParsingException e) {
                e.printStackTrace();
            }
            if(element == null)
                break;
            element.toHtml(this);
        }
        clear();
    }

    public static String escape(String text) {
        return text.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;");
    }

    public abstract void documentEntity(String ref);

    public void appendType(TypeUnparsingContext tuc, Type type, int precedence) {
        StringBuilder b = new StringBuilder();
        type.toString(tuc, b, precedence);
        stringBuilder.append(escape(b.toString()));
    }
    
    public void appendType(TypeUnparsingContext tuc, Type type) {
        appendType(tuc, type, 3);
    }
    
    public void appendRef(String ref) {
        stringBuilder.append("<a name=\"").append(ref).append("\" class=\"ref\">")
                     .append(escape(ref)).append("</a>");
    }

    public Documentation toDocumentation() {
        finishTableOfContents();
        return new Documentation(stringBuilder.toString(), tableOfContents.toString());
    }

    private static final Pattern PARAGRAPH_PATTERN = Pattern.compile("'[^ ']+'|@[^@]+@");
    
    public void appendParagraph(String text) {
        clear();
        stringBuilder.append("<p>");        
        Matcher m = PARAGRAPH_PATTERN.matcher(text);
        int lastAppend = 0;
        while(m.find()) {
            int start = m.start();
            int end = m.end();
            
            stringBuilder.append(text, lastAppend, start);
            char c = text.charAt(start);
            if(c == '\'') {
                String ref = text.substring(start+1, end-1);
                
                stringBuilder.append("<code><a href=\"#").append(ref).append("\">")
                             .append(ref)
                             .append("</a></code>");
            }
            else if(c == '@') {
                stringBuilder.append("<code>").append(text, start+1, end-1).append("</code>");
            }
            lastAppend = end;
        }
        stringBuilder.append(text, lastAppend, text.length());
        stringBuilder.append("</p>\n");        
    }
}
