/*******************************************************************************
 * Copyright (c) 2007, 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - initial selection handling improvements
 *******************************************************************************/
package org.simantics.modeling.ui.pdf;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IFilter;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.requests.Node;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.ui.utils.NoProjectPage;
import org.simantics.project.IProject;
import org.simantics.project.ProjectKeys;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.FileUtils;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;
import org.simantics.utils.ui.workbench.StringMemento;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PDFDiagramExportWizard extends Wizard implements IExportWizard {

    private static final Logger LOGGER = LoggerFactory.getLogger(PDFDiagramExportWizard.class);

    private static final int    MAX_RECENT_EXPORT_PATHS = 10;

    private static final String TAG_PATH  = "path";

    private static final String ATTR_NAME = "name";

    Deque<String>               recentExportPaths;
    boolean                     zoomToFit;
    boolean                     attachTG, attachWiki, addPageNumbers;

    PDFExportPlan               exportPlan;

    private boolean readPreferences() {
        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        String recentPathsPref = store.getString(Preferences.DIAGRAM_EXPORT_PDF_PATH);
        recentExportPaths = decodePaths(recentPathsPref);
        zoomToFit = store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_ZOOM_TO_FIT);
        attachTG =  store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_ATTACH_TG);
        attachWiki =  store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_ATTACH_WIKI);
        addPageNumbers =  store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_ADD_PAGE_NUMBERS);

        return true;
    }

    private void writePreferences() throws IOException {
        IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_PATH, encodePaths(recentExportPaths));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_ZOOM_TO_FIT, String.valueOf(zoomToFit));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_ATTACH_TG, String.valueOf(attachTG));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_ATTACH_WIKI, String.valueOf(attachWiki));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_ADD_PAGE_NUMBERS, String.valueOf(addPageNumbers));

        if (store.needsSaving())
            store.save();
    }

    private Deque<String> decodePaths(String recentPathsPref) {
        Deque<String> result = new LinkedList<>();
        try {
            StringMemento sm = new StringMemento(recentPathsPref);
            for (IMemento m : sm.getChildren(TAG_PATH)) {
                String name = m.getString(ATTR_NAME);
                if (name != null && !name.isEmpty())
                    result.add(name);
            }
        } catch (IllegalArgumentException e) {
        }
        return result;
    }

    private String encodePaths(Deque<String> recentPaths) {
        StringMemento sm = new StringMemento();
        for (String path : recentPaths) {
            IMemento m = sm.createChild(TAG_PATH);
            m.putString(ATTR_NAME, path);
        }
        return sm.toString();
    }

    public PDFDiagramExportWizard() {
        setWindowTitle("Export Diagrams to PDF");
        setNeedsProgressMonitor(true);
    }

    @Override
    public void addPages() {
        super.addPages();
        if (exportPlan != null) {
            addPage(new PDFExportPage(exportPlan));
        } else {
            addPage(new NoProjectPage("Export Diagrams to PDF"));
        }
    }

    private NamedResource toNamedResource(ReadGraph graph, Resource r) throws DatabaseException {
        String name = NameLabelUtil.modalName(graph, r, NameLabelMode.NAME_AND_LABEL);
        return new NamedResource(name, r);
    }

    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        readPreferences();

        ISessionContext ctx = Simantics.getSessionContext();
        if (ctx == null)
            return;
        IProject project = ctx.getHint(ProjectKeys.KEY_PROJECT);
        if (project == null)
            return;

        exportPlan = new PDFExportPlan(ctx, recentExportPaths);
        exportPlan.project = project;
        exportPlan.initialSelection = selection;
        exportPlan.fitContentToPageMargins = zoomToFit;
        exportPlan.attachTG = attachTG;
        exportPlan.attachWiki = attachWiki;
        exportPlan.addPageNumbers = addPageNumbers;
        
        // Get all model names
        try {
            exportPlan.sessionContext.getSession().syncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    Set<Resource> processed = new HashSet<>();
                    List<NamedResource> models = new ArrayList<>();

                    Collection<Resource> ontologies = Simantics.applySCL("Simantics/SharedOntologies", "traverseSharedOntologies", graph, graph.getRootLibrary());
                    for (Resource root : ontologies) {
                        if (processed.add(root))
                            models.add( toNamedResource(graph, root) );
                    }

                    for (Resource model : graph.syncRequest(new ObjectsWithType(exportPlan.project.get(),
                            Layer0.getInstance(graph).ConsistsOf, SimulationResource.getInstance(graph).Model))) {
                        if (processed.add(model))
                            models.add( toNamedResource(graph, model) );
                    }
                    Collections.sort(models, AlphanumComparator.CASE_INSENSITIVE_COMPARATOR);
                    exportPlan.selectableModels.addAll(models);

                    Resource selected = ResourceAdaptionUtils.toSingleResource(selection.getFirstElement());
                    Resource indexRoot = selected != null ? graph.sync(new PossibleIndexRoot(selected)) : null;
                    if (indexRoot != null)
                        exportPlan.initialModelSelection = exportPlan.selection = toNamedResource(graph, indexRoot);

                    if (exportPlan.selection == null && !exportPlan.selectableModels.isEmpty())
                        exportPlan.selection = exportPlan.selectableModels.get(0);
                }
            });
        } catch (DatabaseException e) {
            LOGGER.error("Failed to initialize diagram PDF export wizard input data.", e);
        }
    }

    @Override
    public boolean performFinish() {
        if (exportPlan.exportLocation.exists()) {
            boolean confirmed = MessageDialog.openConfirm(getShell(), "Overwrite", "Are you sure you want to overwrite " + exportPlan.exportLocation);
            if (!confirmed)
                return false;

            try {
                FileUtils.deleteAll(exportPlan.exportLocation);
            } catch (IOException e) {
                ExceptionUtils.logAndShowError(e);
                return false;
            }
        }

        try {
            recentExportPaths.addFirst(exportPlan.exportLocation.getAbsolutePath());

            // Remove duplicates
            Set<String> dups = new TreeSet<>(String.CASE_INSENSITIVE_ORDER);
            for (Iterator<String> it = recentExportPaths.iterator(); it.hasNext();) {
                String path = it.next();
                if (!dups.add(path)) {
                    it.remove();
                }
            }

            if (recentExportPaths.size() > MAX_RECENT_EXPORT_PATHS)
                recentExportPaths.pollLast();

            zoomToFit = exportPlan.fitContentToPageMargins;
            attachTG = exportPlan.attachTG;
            attachWiki = exportPlan.attachWiki;
            addPageNumbers = exportPlan.addPageNumbers;

            writePreferences();
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }

        // Make sure that the diagrams are printed in the same order as the user
        // saw them in the wizard.
        exportPlan.selectedNodes = exportPlan.nodes.depthFirstFlatten(new IFilter() {
            @Override
            public boolean select(Object toTest) {
                Node n = (Node) toTest;
                return exportPlan.selectedNodeSet.contains(n) && n.getDiagramResource() != null;
            }
        }, Node.CASE_INSENSITIVE_COMPARATOR);

        long start = System.currentTimeMillis();
        try {
            getContainer().run(true, true, monitor -> {
                try {
                    DiagramPrinter.printToPdf(monitor, exportPlan, exportPlan.exportLocation.toString(), exportPlan.selectedNodes);
                } catch (PdfException e) {
                    throw new InvocationTargetException(e);
                } finally {
                    monitor.done();
                }
            });
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            ExceptionUtils.logAndShowError(t);
            return false;
        } catch (InterruptedException e) {
            return false;
        }
        long end = System.currentTimeMillis();
        LOGGER.info("PDF export took " + ((end - start) * 1e-3) + " seconds.");

        return true;
    }

}
