/*******************************************************************************
 * Copyright (c) 2007, 2018 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab simantics/platform#133
 *******************************************************************************/
package org.simantics.modeling.ui.modelBrowser.handlers;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.handlers.HandlerUtil;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.CopyHandler;
import org.simantics.db.layer0.util.SimanticsClipboardImpl;
import org.simantics.modeling.ui.Activator;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.ui.SWTUtils;
import org.simantics.utils.ui.workbench.WorkbenchUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardCutHandler extends AbstractHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(StandardCutHandler.class);

    private static IStatusLineManager status;

    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {
        status = WorkbenchUtils.getStatusLine( HandlerUtil.getActiveSite(event) );
        ISelection selection = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getSelectionService().getSelection();
        final Resource[] rs = ResourceAdaptionUtils.toResources( selection );
        if (rs == null) {
            setStatus("Nothing to cut.");
           return null;
        }
        Job job = new Job("Cut resources") {

            @Override
            protected IStatus run(IProgressMonitor monitor) {
                monitor.beginTask("Cut resources", rs.length);
                try {
                    final SimanticsClipboardImpl builder = new SimanticsClipboardImpl();

                    Simantics.getSession().syncRequest(new ReadRequest() {
                        @Override
                        public void run(ReadGraph graph) throws DatabaseException {
                            Set<Resource> unique = new HashSet<>();
                            for (Resource r : rs) {
                                if (!unique.add(r))
                                    continue;
                                CopyHandler handler = graph.adapt(r, CopyHandler.class);
                                handler.cutToClipboard(graph, builder, SubMonitor.convert(monitor, 1));
                            }
                        }
                    });

                    Simantics.setClipboard(builder);
                    setCutMessage(builder.getContents().size(), "resource");
                    return Status.OK_STATUS;
                } catch (DatabaseException e) {
                    LOGGER.error("Cut operation failed", e); //$NON-NLS-1$
                    return new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Cut operation failed", e);
                }
            }
        };
        job.setUser(true);
        job.schedule();

        return null;
    }

    private static void setCutMessage(int count, String elementName) {
        if (count > 1)
            setStatus("Cut " + count + " " + elementName + "s to clipboard");
        else if (count == 1)
            setStatus("Cut " + elementName + " to clipboard");
        else
            setStatus("Nothing to cut.");
    }

    private static void setStatus(String message) {
        if (status != null) {
            SWTUtils.asyncExec(
                    PlatformUI.getWorkbench().getDisplay(),
                    () -> status.setMessage(message));
        }
    }

    public static String cutResourcesToClipboard(final Resource[] rs, ISelection selection) {
        try {
            final SimanticsClipboardImpl builder = new SimanticsClipboardImpl();
            Simantics.getSession().syncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    for (Resource r : rs) {
                        CopyHandler handler = graph.adapt(r, CopyHandler.class);
                        handler.cutToClipboard(graph, builder, null);
                    }
                }
            });
            Simantics.setClipboard(builder);
            setCutMessage(builder.getContents().size(), "resource");
        } catch (DatabaseException e) {
            LOGGER.error("Failed to cut {} resources to clipboard: {}", rs.length, Arrays.toString(rs), e); //$NON-NLS-1$
        }

        return null;
    }

}
