/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor.handlers;

import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.action.IAction;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.g2d.canvas.impl.AbstractCanvasParticipant;
import org.simantics.g2d.canvas.impl.DependencyReflection.Dependency;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.IElement;
import org.simantics.scenegraph.g2d.events.EventHandlerReflection.EventHandler;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDoubleClickedEvent;
import org.simantics.ui.workbench.action.ChooseActionRequest;
import org.simantics.utils.ui.ExceptionUtils;
import org.simantics.utils.ui.action.IPriorityAction;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * Really simple support for activating property view from diagram when a single
 * element is double clicked.
 * 
 * @author Tuukka Lehtonen
 */
public class DefaultElementDoubleClickHandler extends AbstractCanvasParticipant {

    @Dependency
    Selection selection;

    public boolean accept(IElement element) {
    	return true;
    }
    
    @EventHandler(priority = -10)
    public boolean handleDoubleClick(MouseDoubleClickedEvent me) {
    	
        Set<IElement> sel = selection.getSelection(0);
        if (sel.size() == 1) {
        	
            IElement e = sel.iterator().next();
            if(!accept(e)) return false;
            
            final Object data = e.getHint(ElementHints.KEY_OBJECT);

            final Display display = PlatformUI.getWorkbench().getDisplay();
            if (display == null)
                return false;

            // See if there's any highest-priority editor adapter that we could use for the object:
            try {
                Boolean consumed = Simantics.getSession().sync(new UniqueRead<Boolean>() {
                    @Override
                    public Boolean perform(ReadGraph graph) throws DatabaseException {
                        IPriorityAction[] actions = ChooseActionRequest.findActions(graph, null, data, "", false, false);
                        IPriorityAction priorityAction = (IPriorityAction) ChooseActionRequest.chooseAction(null, actions, null, true);
                        if (priorityAction != null && priorityAction.getPriority() >= 10) {
                            //System.out.println("HIGH PRIORITY ACTION(" + priorityAction.getPriority() + "): " + priorityAction);
                            display.asyncExec( actionAsRunnable( priorityAction) );
                            return true;
                        } else if (data instanceof Resource) {
                            PlatformUI.getWorkbench().getDisplay().asyncExec( revealView("org.simantics.browsing.ui.graph.propertyView") );
                            return true;
                        }
                        return false;
                    }
                });
                return consumed;
            } catch (DatabaseException ex) {
                ErrorLogger.defaultLogError(ex);
            }
        }

        return false;
    }

    public static Runnable revealView(final String viewIdPart) {
        return new Runnable() {
            @Override
            public void run() {
                try {
                    WorkbenchUtils.showView(viewIdPart, IWorkbenchPage.VIEW_VISIBLE);
                } catch (PartInitException e) {
                    ExceptionUtils.logAndShowError(e);
                }
            }
        };
    }

    public static Runnable revealAndShowView(final String viewIdPart) {
        return new Runnable() {
            @Override
            public void run() {
                try {
                    WorkbenchUtils.showView(viewIdPart, IWorkbenchPage.VIEW_VISIBLE);

                    // Give the post selection listeners time to settle.
                    PlatformUI.getWorkbench().getDisplay().timerExec( 500, showView(viewIdPart) );
                } catch (PartInitException e) {
                    ExceptionUtils.logAndShowError(e);
                }
            }
        };
    }

    public static Runnable showView(final String viewIdPart) {
        return new Runnable() {
            @Override
            public void run() {
                try {
                    CommandUtil.showView(viewIdPart);
                } catch (ExecutionException e) {
                    ExceptionUtils.logAndShowError(e);
                }
            }
        };
    }

    public static Runnable actionAsRunnable(final IAction action) {
        return new Runnable() {
            @Override
            public void run() {
                action.run();
            }
        };
    }

}
