package org.simantics.modeling.ui.diagram.monitor;

import java.util.Arrays;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchSite;
import org.osgi.framework.Bundle;
import org.simantics.Simantics;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.Combo;
import org.simantics.browsing.ui.swt.widgets.Label;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.browsing.ui.swt.widgets.impl.SelectionListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.common.color.Color;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;
import org.simantics.selectionview.ConfigurationComposite;
import org.simantics.ui.colors.Colors;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.datastructures.Triple;
import org.simantics.utils.ui.BundleUtils;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.SWTUtils;
import org.simantics.utils.ui.gfx.ColorImageDescriptor;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

public class MonitorComposite extends ConfigurationComposite {

    private static final String DATA_CURRENT_COLOR = "COLOR"; //$NON-NLS-1$

    public void create(final Composite body, IWorkbenchSite site, final ISessionContext context, final WidgetSupport support) {
        final Display display = body.getDisplay();

        body.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

        Composite buttonComposite = new Composite(body, SWT.NONE);
        buttonComposite.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().span(2, 1).grab(true, false).applyTo(buttonComposite);
        GridLayoutFactory.fillDefaults().equalWidth(false).numColumns(4).extendedMargins(5,5,5,5).applyTo(buttonComposite);

        LocalResourceManager resourceManager = new LocalResourceManager(JFaceResources.getResources(), buttonComposite);
        support.setParameter(WidgetSupport.RESOURCE_MANAGER, resourceManager);

        Label templateHeader = new Label(buttonComposite, support, 0);
        templateHeader.setText(Messages.MonitorComposite_Template);
        //templateHeader.setFont(smallFont);
        templateHeader.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().grab(false, false).span(1, 1).align(SWT.LEFT, SWT.CENTER).applyTo(templateHeader.getWidget());

        Combo templateCombo = new Combo(buttonComposite, support, SWT.NONE | SWT.READ_ONLY);
        templateCombo.setItemFactory(new AvailableTemplateFactory());
        templateCombo.setSelectionFactory(new CurrentTemplateFactory());
        templateCombo.addModifyListener(new TemplateModifier());
        GridDataFactory.fillDefaults().grab(true, false).applyTo(templateCombo.getWidget());

        Button setDefaultButton = new Button(buttonComposite, support, SWT.NONE | SWT.READ_ONLY);
        setDefaultButton.setText(Messages.MonitorComposite_SetAsDefault);
        setDefaultButton.setTooltipText(Messages.MonitorComposite_SetAsDefaultForNewlyCreatedMonitors);
        setDefaultButton.addSelectionListener(new SelectionListenerImpl<Resource>(context) {
            @Override
            public void apply(WriteGraph graph, Resource monitor) throws DatabaseException {
                graph.markUndoPoint();
                Layer0X L0X = Layer0X.getInstance(graph);
                DiagramResource DIA = DiagramResource.getInstance(graph);
                Resource root = graph.syncRequest(new PossibleIndexRoot(monitor));
                Resource currentTemplate = graph.getPossibleObject(monitor, L0X.ObtainsProperty);
                if (root != null && !graph.isImmutable(root) && currentTemplate != null) {
                    graph.deny(root, DIA.HasDefaultMonitorTemplate);
                    graph.claim(root, DIA.HasDefaultMonitorTemplate, currentTemplate);
                }
            }
        });
        GridDataFactory.fillDefaults().grab(false, false).applyTo(setDefaultButton.getWidget());

        Button resetButton = new Button(buttonComposite, support, SWT.NONE | SWT.READ_ONLY);
        resetButton.setText(Messages.MonitorComposite_ResetLocalChanges);
        resetButton.addSelectionListener(new SelectionListenerImpl<Resource>(context) {
            @Override
            public void apply(WriteGraph graph, Resource monitor) throws DatabaseException {
                graph.markUndoPoint();
                Layer0X L0X = Layer0X.getInstance(graph);
                DiagramResource DIA = DiagramResource.getInstance(graph);
                if(graph.hasStatement(monitor, DIA.HasFont)) graph.deny(monitor, DIA.HasFont);
                if(graph.hasStatement(monitor, DIA.HasFormatter)) graph.deny(monitor, DIA.HasFormatter);
                if(graph.hasStatement(monitor, DIA.HasColor)) graph.deny(monitor, DIA.HasColor);
                if(graph.hasStatement(monitor, L0X.HasExpression)) graph.deny(monitor, L0X.HasExpression);
            }
        });

        GridDataFactory.fillDefaults().grab(false, false).applyTo(resetButton.getWidget());

        Label fontHeader = new Label(buttonComposite, support, 0);
        fontHeader.setText(Messages.MonitorComposite_FontFamily);
        //fontHeader.setFont(smallFont);
        fontHeader.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().grab(false, false).span(1, 1).align(SWT.LEFT, SWT.CENTER).applyTo(fontHeader.getWidget());

        Combo fontCombo = new Combo(buttonComposite, support, SWT.NONE | SWT.READ_ONLY);
        fontCombo.setItemFactory2(new AvailableFontFactory());
        fontCombo.setSelectionFactory(new CurrentFontFactory());
        fontCombo.addModifyListener(new FontModifier());
        GridDataFactory.fillDefaults().grab(true, false).span(3,1).applyTo(fontCombo.getWidget());

        Label sizeHeader = new Label(buttonComposite, support, 0);
        sizeHeader.setText(Messages.MonitorComposite_FontSize);
        //sizeHeader.setFont(smallFont);
        sizeHeader.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().grab(false, false).span(1, 1).align(SWT.LEFT, SWT.CENTER).applyTo(sizeHeader.getWidget());

        Combo sizeCombo = new Combo(buttonComposite, support, SWT.NONE | SWT.READ_ONLY);
        sizeCombo.setItemFactory2(new AvailableFontSizeFactory());
        sizeCombo.setSelectionFactory(new CurrentFontSizeFactory());
        sizeCombo.addModifyListener(new FontSizeModifier());
        GridDataFactory.fillDefaults().grab(false, false).span(3,1).minSize(50, 0).applyTo(sizeCombo.getWidget());

        Label formatterHeader = new Label(buttonComposite, support, 0);
        formatterHeader.setText(Messages.MonitorComposite_Formatting);
        //formatterHeader.setFont(smallFont);
        formatterHeader.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().grab(false, false).span(1, 1).align(SWT.LEFT, SWT.CENTER).applyTo(formatterHeader.getWidget());

        Combo formatCombo = new Combo(buttonComposite, support, SWT.NONE | SWT.READ_ONLY);
        formatCombo.setItemFactory(new AvailableFormatFactory());
        formatCombo.setSelectionFactory(new CurrentFormatFactory());
        formatCombo.addModifyListener(new FormatModifier());
        GridDataFactory.fillDefaults().grab(true, false).span(3,1).applyTo(formatCombo.getWidget());

        Composite rowComposite = new Composite(buttonComposite, SWT.NONE);
        rowComposite.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().span(3, 1).grab(true, false).applyTo(rowComposite);
        GridLayoutFactory.fillDefaults().equalWidth(false).numColumns(4).extendedMargins(5,5,5,5).applyTo(rowComposite);

        Label alignHeader = new Label(rowComposite, support, 0);
        alignHeader.setText(Messages.MonitorComposite_Alignment);
        //alignHeader.setFont(smallFont);
        alignHeader.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().grab(false, false).span(1, 1).align(SWT.LEFT, SWT.CENTER).applyTo(alignHeader.getWidget());

        Bundle iconBundle = Platform.getBundle("com.famfamfam.silk"); //$NON-NLS-1$

        Composite alignComposite = new Composite(rowComposite, SWT.NONE);
        alignComposite.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().span(1, 1).grab(false, false).applyTo(alignComposite);
        GridLayoutFactory.fillDefaults().equalWidth(false).numColumns(4).extendedMargins(5,5,5,5).applyTo(alignComposite);

        Button leadButton = new Button(alignComposite, support, SWT.TOGGLE);
        leadButton.setTooltipText(Messages.MonitorComposite_LeftAlignment);
        leadButton.setImage((Image) resourceManager.get(BundleUtils.getImageDescriptorFromBundle(iconBundle, "icons/text_align_left.png"))); //$NON-NLS-1$
        leadButton.setSelectionFactory(new AlignmentSelectedFactory(G2DResource.URIs.Alignment_Leading));
        leadButton.addSelectionListener(new AlignmentSelectionListener(context, G2DResource.URIs.Alignment_Leading));

        Button centerButton = new Button(alignComposite, support, SWT.TOGGLE);
        centerButton.setTooltipText(Messages.MonitorComposite_CenterAlignment);
        centerButton.setImage((Image) resourceManager.get(BundleUtils.getImageDescriptorFromBundle(iconBundle, "icons/text_align_center.png"))); //$NON-NLS-1$
        centerButton.setSelectionFactory(new AlignmentSelectedFactory(G2DResource.URIs.Alignment_Center));
        centerButton.addSelectionListener(new AlignmentSelectionListener(context, G2DResource.URIs.Alignment_Center));

        Button trailButton = new Button(alignComposite, support, SWT.TOGGLE);
        trailButton.setTooltipText(Messages.MonitorComposite_RightAlignment);
        trailButton.setImage((Image) resourceManager.get(BundleUtils.getImageDescriptorFromBundle(iconBundle, "icons/text_align_right.png"))); //$NON-NLS-1$
        trailButton.setSelectionFactory(new AlignmentSelectedFactory(G2DResource.URIs.Alignment_Trailing));
        trailButton.addSelectionListener(new AlignmentSelectionListener(context, G2DResource.URIs.Alignment_Trailing));

        Label colorHeader = new Label(rowComposite, support, 0);
        colorHeader.setText(Messages.MonitorComposite_Color);
        colorHeader.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().indent(20, 0).grab(false, false).span(1, 1).align(SWT.LEFT, SWT.CENTER).applyTo(colorHeader.getWidget());

        Button colorButton = new Button(rowComposite, support, SWT.PUSH);
        colorButton.setImageFactory( new CurrentColorImageFactory(colorButton.getWidget(), DATA_CURRENT_COLOR) );
        GridDataFactory.fillDefaults().indent(0, 0).grab(false, false).span(1, 1).align(SWT.LEFT, SWT.CENTER).applyTo(colorButton.getWidget());
        StyleEditWidget colorWidget = new StyleEditWidget(colorButton.getWidget());
        colorButton.addSelectionListener(colorWidget);
        support.register(colorWidget);
    }

    static class StyleEditWidget implements Widget, SelectionListener {

        Control control;
        Object input;

        public StyleEditWidget(Control control) {
            this.control = control;
        }

        @Override
        public void widgetSelected(SelectionEvent e) {
            final Resource[] resources = ResourceAdaptionUtils.toResources(input);
            if (resources.length != 0) {
                ColorDialog dialog = new ColorDialog(WorkbenchUtils.getActiveWorkbenchWindowShell(), SWT.NONE);
                dialog.setText(Messages.MonitorComposite_SetMonitorColor);
                Color oldColor = (Color) control.getData(DATA_CURRENT_COLOR);
                if (oldColor != null) {
                    RGB oldRgb = Colors.rgb(oldColor);
                    dialog.setRGB(oldRgb);
                }
                RGB rgb = dialog.open();
                if (rgb != null) {
                    setColor(resources, rgb);
                }
            }
        }

        @Override
        public void widgetDefaultSelected(SelectionEvent e) {
            widgetSelected(e);
        }

        @Override
        public void setInput(ISessionContext context, Object input) {
            this.input = input;
        }

    }

    static class AlignmentSelectedFactory extends ReadFactoryImpl<Resource, Boolean> {
        private String alignmentURI;

        public AlignmentSelectedFactory(String alignmentURI) {
            this.alignmentURI = alignmentURI;
        }

        @Override
        public Object getIdentity(Object inputContents) {
            return new Triple<Object, Object, Class<?>>(inputContents, alignmentURI, getClass());
        }

        @Override
        public Boolean perform(ReadGraph graph, Resource monitor) throws DatabaseException {
            G2DResource G2D = G2DResource.getInstance(graph);
            Resource expectedAlignment = graph.getResource(alignmentURI);
            Resource alignment = graph.getPossibleObject(monitor, G2D.HasHorizontalAlignment);
            return ObjectUtils.objectEquals(expectedAlignment, alignment);
        }
    }

    static class AlignmentSelectionListener extends SelectionListenerImpl<Resource> {
        private String alignmentURI;

        public AlignmentSelectionListener(ISessionContext context, String alignmentURI) {
            super(context);
            this.alignmentURI = alignmentURI;
        }

        @Override
        public void apply(WriteGraph graph, Resource monitor) throws DatabaseException {
            graph.markUndoPoint();
            G2DResource G2D = G2DResource.getInstance(graph);
            Resource alignment = graph.getResource(alignmentURI);
            graph.deny(monitor, G2D.HasHorizontalAlignment);
            graph.claim(monitor, G2D.HasHorizontalAlignment, null, alignment);
        }
    }

    private static void setColor(final Resource[] resources, final RGB rgb) {
        Color c = Colors.irgb(rgb);
        final double[] color = new double[] { c.getR(), c.getG(), c.getB() };
        try {
            Simantics.sync(new WriteRequest() {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    graph.markUndoPoint();
                    Layer0 L0 = Layer0.getInstance(graph);
                    DiagramResource DIA = DiagramResource.getInstance(graph);
                    for (Resource r : resources) {
                        Resource realizedColor = graph.syncRequest(new PossibleObjectWithType(r, DIA.HasColor, DIA.RealizedColor));
                        if (realizedColor == null) {
                            realizedColor = graph.newResource();
                            graph.claim(realizedColor, L0.InstanceOf, null, DIA.RealizedColor);
                            graph.claim(r, DIA.HasColor, realizedColor);
                        }
                        graph.claimLiteral(realizedColor, DIA.RealizedColor_HasRGB, L0.DoubleArray, color, Bindings.DOUBLE_ARRAY);
                    }
                    CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                    graph.addMetadata(cm.add("Set color to " + rgb + " for resources " + Arrays.toString(resources))); //$NON-NLS-1$ //$NON-NLS-2$
                }
            });
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }
    }

    private static class CurrentColorImageFactory extends ReadFactoryImpl<Resource, ImageDescriptor> {

        Control control;
        String colorKey;

        public CurrentColorImageFactory(Control control, String colorKey) {
            this.control = control;
            this.colorKey = colorKey;
        }

        @Override
        public ImageDescriptor perform(ReadGraph graph, Resource monitor) throws DatabaseException {
            DiagramResource DIA = DiagramResource.getInstance(graph);
            Color c = graph.getPossibleRelatedAdapter(monitor, DIA.HasColor, Color.class);
            if (c == null)
                 c = new org.simantics.datatypes.literal.RGB.Integer(0, 0, 0);
            final Color color = c;
            if (control != null) {
                SWTUtils.asyncExec(control, new Runnable() {
                    @Override
                    public void run() {
                        if (!control.isDisposed())
                            control.setData(colorKey, color);
                    }
                });
            }
            RGB rgb = Colors.rgb(color);
            return new ColorImageDescriptor(rgb.red, rgb.green, rgb.blue, 16, 16, false);
        }

    }

}
