package org.simantics.modeling.ui.actions.style;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;

import org.eclipse.osgi.util.NLS;
import org.simantics.utils.strings.format.MetricsFormat;
import org.simantics.utils.strings.format.MetricsFormatList;
import org.simantics.utils.strings.format.MetricsFormatListListener;

public class MetricsEditor extends JPanel {

    private static final long serialVersionUID = 3781280603509654789L;
    private JTextField valueTestField;
    private JTextField valuePresentationField;
    private JTextField formatNameField;
    private JTextField formatPatternField;
    //		private JTextField formatScaleField;

    private MetricsFormat format;
    private MetricsFormatList formatList;
    private double formatValue;

    private JTable formatTable;
    private MetricsTableModel tableModel;

    public MetricsEditor() {
        super();
        setLayout(new BorderLayout());

        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        valueTestField = new JTextField();
        valuePresentationField = new JTextField();
        formatNameField = new JTextField();
        formatPatternField = new JTextField();
        //			formatScaleField = new JTextField();

        JPanel labelPanel = new JPanel(new GridLayout(0,1));
        JPanel fieldPanel = new JPanel(new GridLayout(0,1));
        panel.add(labelPanel,BorderLayout.WEST);
        panel.add(fieldPanel,BorderLayout.CENTER);


        labelPanel.add(new JLabel(Messages.MetricsEditor_ValueTest));
        fieldPanel.add(valueTestField);
        labelPanel.add(new JLabel(Messages.MetricsEditor_ValuePresentation));
        fieldPanel.add(valuePresentationField);
        labelPanel.add(new JLabel(Messages.MetricsEditor_FormatName));
        fieldPanel.add(formatNameField);
        labelPanel.add(new JLabel(Messages.MetricsEditor_FormatPattern));
        fieldPanel.add(formatPatternField);

        add(panel,BorderLayout.NORTH);
        //			add(new JLabel("Format scale:"));
        //			add(formatScaleField);

        JPanel listPanel = new JPanel();
        add(listPanel,BorderLayout.CENTER);
        JTable table = createFormatTable();
        JScrollPane scrollPane = new JScrollPane(table);
        table.setFillsViewportHeight(true);

        listPanel.add(scrollPane);

        JPanel controlPanel = new JPanel();
        add(controlPanel,BorderLayout.SOUTH);
        controlPanel.setLayout(new FlowLayout(FlowLayout.LEFT));

        JButton addTemplateButton = new JButton(Messages.MetricsEditor_AddFormatTemplate);
        addTemplateButton.setToolTipText(Messages.MetricsEditor_AddFormatTemplateTT);
        controlPanel.add(addTemplateButton);
        addTemplateButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent arg0) {
                addTemplate();
            }
        });

        JButton removeTemplateButton = new JButton(Messages.MetricsEditor_RemoveTemplate);
        removeTemplateButton.setToolTipText(Messages.MetricsEditor_RemoveTemplateTT);
        controlPanel.add(removeTemplateButton);
        removeTemplateButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent arg0) {
                removeTemplate();
            }
        });

        JButton updateTemplateButton = new JButton(Messages.MetricsEditor_UpdateTemplate);
        updateTemplateButton.setToolTipText(Messages.MetricsEditor_UpdateTemplateTT);
        controlPanel.add(updateTemplateButton);
        updateTemplateButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent arg0) {
                updateTemplate();
            }
        });

        JButton newFormatButton = new JButton(Messages.MetricsEditor_NewFormat);
        newFormatButton.setToolTipText(Messages.MetricsEditor_NewFormatTT);
        controlPanel.add(newFormatButton);
        newFormatButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent arg0) {
                newFormat();
            }
        });


        valuePresentationField.setEditable(false);

        ActionListener l = new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                updateValues();

            }
        };

        valueTestField.addActionListener(l);
        formatPatternField.addActionListener(l);

        valueTestField.setText(Messages.MetricsEditor_Numbers);
        formatPatternField.setText("%s"); //$NON-NLS-1$

        updateValues();
    }

    private void addTemplate() {
        MetricsFormat format = createMetricsFormatFromFields();

        try {
            formatList.addFormat(format);
        } catch (Exception e) {
            // TODO : show error
        }
    }

    private void updateTemplate() {
        int index = formatTable.getSelectedRow();
        if (index == -1)
            return; // TODO : show error, or even better, disable the button
        MetricsFormat format = createMetricsFormatFromFields();
        try {
            formatList.replaceFormatWith(formatList.getFormats()[index], format);
        } catch (Exception e) {
            // TODO : show error
        }
    }

    private void removeTemplate() {
        int index = formatTable.getSelectedRow();
        if (index == -1)
            return; // TODO : show error, or even better, disable the button
        try {
            formatList.removeFormat(formatList.getFormats()[index]);
        } catch (Exception e) {
            // TODO : show error
        }
    }

    public MetricsFormat getFormat() {
        return format;
    }

    private void newFormat() {
        format = null;
        formatNameField.setText(Messages.MetricsEditor_NewFormatField);
        formatPatternField.setText("%s"); //$NON-NLS-1$
        updateValues();
    }

    private JTable createFormatTable() {
        formatList = MetricsFormatList.getList();
        tableModel = new MetricsTableModel(formatList);

        formatTable = new JTable(tableModel);
        formatTable.doLayout();

        formatTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        formatTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {

            @Override
            public void valueChanged(ListSelectionEvent e) {
                if (e.getValueIsAdjusting())
                    return;
                int index = formatTable.getSelectionModel().getMinSelectionIndex();
                if (index == -1)
                    return;
                setMetricsFormat(formatList.getFormats()[index]);

            }
        });
        return formatTable;
    }

    private void updateValues() {
        String value = valueTestField.getText();
        if (value == null || value.length() == 0)
            return;
        Double d = null;
        try {
            d = Double.parseDouble(value);
        } catch (NumberFormatException e) {
            valuePresentationField.setText(Messages.MetricsEditor_ValueTestNotANumber);
            return;
        }
        formatValue = d;

        MetricsFormat format = null;
        try {
            format = createMetricsFormatFromFields();
        } catch (Exception e) {
            valuePresentationField.setText(NLS.bind(Messages.MetricsEditor_FormatPatternNotCorrect, e.getMessage()));
        }
        if (format == null)
            return; // TODO : show error
        setMetricsFormat(format);
        tableModel.setFormatValue(formatValue);
    }

    private MetricsFormat createMetricsFormatFromFields() {
        if (formatPatternField.getText().length() == 0)
            return null;
        return new MetricsFormat(formatPatternField.getText(),1.0,formatNameField.getText());
    }

    public void setMetricsFormat(MetricsFormat format) {
        this.format = format;
        try {
            valuePresentationField.setText(format.formatValue(formatValue));
        } catch (Exception e) {
            valuePresentationField.setText(NLS.bind(Messages.MetricsEditor_FormatError, e.getMessage()));
        }
        formatNameField.setText(format.getName());
        formatPatternField.setText(format.getPattern());
    }

    public void dispose() {
        tableModel.dispose();
    }

    private class MetricsTableModel implements TableModel, MetricsFormatListListener {

        private String[] columnNames = {Messages.MetricsEditor_Name,Messages.MetricsEditor_Format,Messages.MetricsEditor_Value};
        private MetricsFormatList formatList;
        private double formatValue = 0;

        public MetricsTableModel(MetricsFormatList formatList) {
            this.formatList = formatList;
            this.formatList.addListener(this);
        }

        public void setFormatValue(double value) {
            this.formatValue = value;
            fireChanged();
        }

        @Override
        public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
            // do nothing
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return false;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            MetricsFormat format = formatList.getFormats()[rowIndex];
            if (columnIndex == 0) {
                return format.getName();
            } else if (columnIndex == 1) {
                return format.getPattern();
            } else if (columnIndex == 2) {
                return format.formatValue(formatValue);
            }
            throw new IndexOutOfBoundsException("There is no column " + columnIndex); //$NON-NLS-1$
        }

        @Override
        public int getRowCount() {
            return formatList.getFormats().length;
        }

        @Override
        public String getColumnName(int columnIndex) {
            return columnNames[columnIndex];
        }

        @Override
        public int getColumnCount() {
            return columnNames.length;
        }

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return String.class;
        }

        private List<TableModelListener> listeners = new ArrayList<TableModelListener>();

        @Override
        public void addTableModelListener(TableModelListener l) {
            listeners.add(l);
        }

        @Override
        public void removeTableModelListener(TableModelListener l) {
            listeners.remove(l);
        }

        @Override
        public void onListChanged(MetricsFormatList sender) {
            fireChanged();
        }

        private void fireChanged() {
            for (TableModelListener l : listeners) {
                l.tableChanged(new TableModelEvent(this));
            }
        }

        public void dispose() {
            formatList.removeListener(this);
        }
    }
}
