package org.simantics.modeling.adapters;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.simantics.utils.strings.EString;

/**
 * @author Tuukka Lehtonen
 */
public class RemoveInstancesDialog extends MessageDialog {

	public static class Content {
		public String label;
		public ImageDescriptor image;
		public String details;

		public Content(String label) {
			assert label != null;
			this.label = label;
		}

		@Override
		public String toString() {
			if (details == null)
				return label;
			return label + "\nDetails: " + details;
		}
	}

	protected String message;
	protected boolean hasDetails; 

	protected Content[] content;

	protected LocalResourceManager resourceManager;
	protected Composite tableAndButtons;
	protected Text messageText;
	protected Text detailsText;
	private GridData detailsGridData;

	public RemoveInstancesDialog(Shell parentShell, String dialogTitle, String dialogMessage, Content[] content) {
		this(parentShell,
				dialogTitle,
				dialogMessage,
				MessageDialog.QUESTION,
				new String[] {
					IDialogConstants.PROCEED_LABEL,
					IDialogConstants.CANCEL_LABEL
				},
				1,
				content);
	}

	public RemoveInstancesDialog(Shell parentShell, String dialogTitle, String dialogMessage, int dialogType, String[] buttons, int defaultIndex, Content[] content) {
		super(parentShell,
				dialogTitle,
				null,
				null,
				dialogType,
				buttons,
				defaultIndex);

		this.message = dialogMessage;
		this.content = content;
		this.hasDetails = contentHasDetails(content);
	}

	private boolean contentHasDetails(Content[] content) {
		for (Content c : content) {
			if (c.details != null && !c.details.trim().isEmpty())
				return true;
		}
		return false;
	}

	@Override
	protected int getShellStyle() {
		return super.getShellStyle() | SWT.RESIZE;
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		resourceManager = new LocalResourceManager(JFaceResources.getResources(), parent);
		Composite composite = (Composite) super.createDialogArea(parent);
		return composite;
	}

	@Override
	protected Control createMessageArea(Composite composite) {
		super.createMessageArea(composite);
		// create message
		if (message != null) {
			messageText = new Text(composite, getMessageLabelStyle());
			messageText.setText(message);
			messageText.setEditable(false);
			GridDataFactory
					.fillDefaults()
					.align(SWT.FILL, SWT.BEGINNING)
					.grab(true, false)
					.hint(
							convertHorizontalDLUsToPixels(IDialogConstants.MINIMUM_MESSAGE_AREA_WIDTH),
							SWT.DEFAULT).applyTo(messageText);
		}
		return composite;
	}

	@Override
	protected Control createCustomArea(Composite parent) {
		tableAndButtons = new Composite(parent, 0);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(tableAndButtons);
		GridLayoutFactory.fillDefaults().numColumns(2).applyTo(tableAndButtons);

		Label uses = new Label(tableAndButtons, 0);
		uses.setText("Instances:");
		GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(uses);

		TableColumnLayout ad = new TableColumnLayout();
		Composite tableComposite = new Composite(tableAndButtons, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, true).minSize(300, 300).applyTo(tableComposite);
		tableComposite.setLayout(ad);
		TableViewer viewer = new TableViewer(tableComposite, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER | SWT.FULL_SELECTION);
		viewer.setContentProvider(new CP());
		TableViewerColumn c1 = new TableViewerColumn(viewer, SWT.LEFT);
		c1.setLabelProvider(new LP());
		ad.setColumnData(c1.getColumn(), new ColumnWeightData(100, 100, true));

		Composite buttons = new Composite(tableAndButtons, 0);
		GridDataFactory.fillDefaults().grab(false, true).applyTo(buttons);
		GridLayoutFactory.fillDefaults().numColumns(1).applyTo(buttons);

		Button copyButton = new Button(buttons, SWT.PUSH);
		copyButton.setText("&Copy to Clipboard");
		copyButton.setToolTipText("Copy List Contents to Clipboard");
		copyButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				String text = EString.implode(content, String.format("%n"));
				Clipboard cp = new Clipboard(e.display);
				cp.setContents(new Object[] { text }, new Transfer[] { TextTransfer.getInstance() });
				cp.dispose();
			}
		});

//		Button navigateButton = new Button(buttons, SWT.PUSH);
//		navigateButton.setText("Go To");
//		navigateButton.setToolTipText("Go To Selected Instances");

		detailsText = new Text(tableAndButtons, SWT.BORDER | SWT.WRAP | SWT.MULTI | SWT.V_SCROLL);
		Color bg = detailsText.getBackground();
		detailsText.setEditable(false);
		detailsText.setBackground(bg);
		detailsGridData = GridDataFactory.fillDefaults().grab(true, false).hint(SWT.DEFAULT, 100).span(2, 1).create();
		detailsText.setLayoutData(detailsGridData);

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection sel = (IStructuredSelection) event.getSelection();
				Content content = (Content)(sel.getFirstElement());
				showDetails(content);
			}
		});
		viewer.setInput(content);

		detailsText.getDisplay().asyncExec(new Runnable() {
			@Override
			public void run() {
				showDetails(null);
			}
		});
		return tableAndButtons;
	}

	protected void showDetails(Content content) {
		if (detailsText == null || detailsText.isDisposed())
			return;
		if (content != null && content.details != null) {
			detailsText.setText(content.details);
			if (detailsGridData.exclude) {
				detailsGridData.exclude = false;
				detailsGridData.heightHint = 100;
				tableAndButtons.layout(true, true);
			}
		} else {
			detailsText.setText("");
			if (!detailsGridData.exclude) {
				detailsGridData.exclude = true;
				detailsGridData.heightHint = 0;
				detailsText.setSize(0, 0);
				tableAndButtons.layout(true, true);
			}
		}
	}

	protected class CP implements IStructuredContentProvider {

		@Override
		public void dispose() {
		}

		@Override
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		@Override
		public Object[] getElements(Object inputElement) {
			return content;
		}

	}

	protected class LP extends ColumnLabelProvider {
		@Override
		public Image getImage(Object element) {
			Content content = (Content) element;
			return content.image != null ? (Image) resourceManager.get(content.image) : null;
		}
		@Override
		public String getText(Object element) {
			Content content = (Content) element;
			return content.label;
		}
	}

}
