package org.simantics.help.ui;

import java.util.function.Supplier;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.simantics.Simantics;
import org.simantics.db.Session;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.combinations.Combinators;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.ui.workbench.ResourceEditorSupport;
import org.simantics.ui.workbench.TitleUpdater;
import org.simantics.ui.workbench.TitleWithParentNameRequest;
import org.simantics.ui.workbench.ToolTipRequest;
import org.simantics.ui.workbench.editor.input.InputValidationCombinators;

import winterwell.markdown.editors.MarkdownEditor;

public class HelpFileEditor extends MarkdownEditor {

    private static final String EDITOR_ID = "org.simantics.help.ui.HelpFileEditor"; //$NON-NLS-1$

    private boolean disposed;

    private static final ParametrizedRead<IResourceEditorInput, Boolean> INPUT_VALIDATOR = 
            Combinators.compose(InputValidationCombinators.hasURI(), InputValidationCombinators.extractInputResource());

    private ResourceEditorSupport support;

    public HelpFileEditor() {
        super();
        setDocumentProvider(new HelpFileDocumentProvider());
    }

    @Override
    public void init(IEditorSite site, IEditorInput input) throws PartInitException {
        super.init(site, input);

        support = new ResourceEditorSupport(this, INPUT_VALIDATOR);
        support.activateValidation();
        try {
            getResourceInput().init(null);
        } catch (DatabaseException e) {
            throw new PartInitException("Failed to initialize " + input, e); //$NON-NLS-1$
        }

    }

    protected IResourceEditorInput getResourceInput() {
        return (IResourceEditorInput) getEditorInput();
    }

    @Override
    public void createPartControl(Composite parent) {
        super.createPartControl(parent);
        updatePartName();
    }

    protected void updatePartName() {
        setPartName(getEditorInput().getName());

        Session session = Simantics.peekSession();
        if (session != null) {
            Supplier<Boolean> disposedCallback = this::isDisposed;
            session.asyncRequest(new TitleWithParentNameRequest(getResourceInput()),
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setPartName, disposedCallback));
            session.asyncRequest(new ToolTipRequest(getSite().getId(), getResourceInput()),
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setTitleToolTip, disposedCallback));
        }
    }

    @Override
    public void dispose() {
        disposed = true;
        if (support != null) {
            support.dispose();
            support = null;
        }
        super.dispose();
    }

    public boolean isDisposed() {
        return disposed;
    }

    public static String getEditorId() {
        return EDITOR_ID;
    }
}
