/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphviz;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.HashMap;

import javax.swing.SwingUtilities;

import org.simantics.graphviz.continuation.Computation;
import org.simantics.graphviz.drawable.GraphDrawable;
import org.simantics.graphviz.drawable.JViewer;
import org.simantics.graphviz.internal.parser.DotParser;
import org.simantics.graphviz.internal.parser.ParseException;
import org.simantics.graphviz.internal.process.CreateXDot;

/**
 * A class containing only static methods for operating
 * on graphs.
 * 
 * @author Hannu Niemist�
 */
public class Graphs {

    private Graphs() {}
    
    /**
     * Parses a dot graph from an input stream.
     */
    public static Graph parse(InputStream stream) throws IOException {
        try {
            return new DotParser(stream).document();
        } catch(ParseException e) {
        	System.out.println(
        			e.currentToken.beginLine + ":" + e.currentToken.beginColumn + " - " +
        			e.currentToken.endLine + ":" + e.currentToken.endColumn);
            throw new IOException(e);
        }
    }
    
    /**
     * Parses a dot graph from a file.
     */
    public static Graph parse(File file) throws IOException {
        InputStream stream = new FileInputStream(file);
        Graph graph = parse(stream);
        stream.close();
        
        return graph;
    }
    
    /**
     * Layouts a graph with a given algorithm and writes it to the file using given output format.
     */
    public static void createImage(Graph graph, String algorithm, String type, File file) throws IOException {
        System.out.println("Create image");
        if(type.equals("dot")) {
            PrintStream stream = new PrintStream(new FileOutputStream(file));
            graph.write(stream);
            stream.close();
        }
        else {  
        	File DOT_EXE = Activator.getDotExe();
        	Process process = new ProcessBuilder(
                  DOT_EXE.toString(), "-T" + type, "-K" + algorithm, "-o"+ file.getAbsolutePath())
                  .directory(DOT_EXE.getParentFile())
                  .start();
            
            PrintStream stream = new PrintStream(process.getOutputStream());
            
            graph.write(stream);
            stream.close();
           
            InputStream errorStream = process.getErrorStream();
            while(true) {
                int c = errorStream.read();
                if(c <= 0)
                    break;
                System.err.print((char)c);
            }
            errorStream.close();
            
            try {
                process.waitFor();
            } catch (InterruptedException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }
    
    /**
     * Layouts the graph.
     */
    public static Computation<Graph> createXDot(Graph graph, String algorithm) {
        CreateXDot process = new CreateXDot(graph, algorithm);
        process.start();
        return process;
    }
    
    public static byte[] read(InputStream stream) throws IOException {
    	byte[] buffer = new byte[1024];
    	int pos = 0;
    	while(true) {    		
    		int len = stream.read(buffer, pos, buffer.length-pos);
    		if(len < 0)
    			return Arrays.copyOf(buffer, pos);
    		pos += len;
    		if(pos == buffer.length) {
    			buffer = Arrays.copyOf(buffer, (int)(buffer.length * 1.5));
    		}
    	}
    }

    /**
     * Layouts the graph using given algorithm and visualizes it
     * in a new viewer window.
     */
    public static void show(final Graph graph, final String algorithm) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                new JViewer(new GraphDrawable(graph, algorithm));
            }
        });        
    }   
    
    /**
     * Layouts the graph using dot algorithm and visualizes it
     * in a new viewer window.
     */
    public static void show(final Graph graph) {
        show(graph, "dot");      
    }   
    
    public static void showWithNamedWindow(final String windowName, final Graph graph, final String algorithm) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                JViewer.getOrCreateViewer(windowName, new GraphDrawable(graph, algorithm));
            }
        });
    }
    
}
