/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.routing.algorithm1;

import java.awt.geom.Rectangle2D;


public interface Direction {
	int getId();
	double front(Rectangle rectangle);
	double back(Rectangle rectangle);
	double minSide(Rectangle rectangle);
	double maxSide(Rectangle rectangle);
	double front(Rectangle2D rectangle);
	double back(Rectangle2D rectangle);
	double minSide(Rectangle2D rectangle);
	double maxSide(Rectangle2D rectangle);
	Direction turnLeft();
	Direction turnRight();
	double getDir(double x, double y);
	double getSide(double x, double y);
	RoutePencil createPencil(double front, double back, double min, double max, double penalty, RoutePencil parent);
	
	public static final Direction EAST = new East();
	public static final Direction SOUTH = new South();
	public static final Direction WEST = new West();
	public static final Direction NORTH = new North();
	public static final Direction[] directions = new Direction[] { EAST, SOUTH, WEST, NORTH };
	
	public static class East implements Direction {
		public int getId() { return 0; }
		public double front(Rectangle rect) { return rect.x0; }
		public double back(Rectangle rect) { return rect.x1; }
		public double minSide(Rectangle rect) { return rect.y0; }
		public double maxSide(Rectangle rect) { return rect.y1; }
		public double front(Rectangle2D rect) { return rect.getMinX(); }
		public double back(Rectangle2D rect) { return rect.getMaxX(); }
		public double minSide(Rectangle2D rect) { return rect.getMinY(); }
		public double maxSide(Rectangle2D rect) { return rect.getMaxY(); }
		public Direction turnLeft() { return NORTH; }
		public Direction turnRight() { return SOUTH; }		
		public RoutePencil createPencil(double front, double back, double min, double max, double penalty, RoutePencil parent) {
			return new RoutePencil(front, min, back, max, penalty, this, parent);
		}		
		public double getDir(double x, double y) { return x; }	
		public double getSide(double x, double y) { return y; }
	}
	
	public static class South implements Direction {
		public int getId() { return 1; }
		public double front(Rectangle rect) { return rect.y0; }
		public double back(Rectangle rect) { return rect.y1; }
		public double minSide(Rectangle rect) { return rect.x0; }
		public double maxSide(Rectangle rect) { return rect.x1; }
		public double front(Rectangle2D rect) { return rect.getMinY(); }
		public double back(Rectangle2D rect) { return rect.getMaxY(); }
		public double minSide(Rectangle2D rect) { return rect.getMinX(); }
		public double maxSide(Rectangle2D rect) { return rect.getMaxX(); }
		public Direction turnLeft() { return EAST; }
		public Direction turnRight() { return WEST; }		
		public RoutePencil createPencil(double front, double back, double min, double max, double penalty, RoutePencil parent) {
			return new RoutePencil(min, front, max, back, penalty, this, parent);
		}		
		public double getDir(double x, double y) { return y; }	
		public double getSide(double x, double y) { return x; }
	}
	
	public static class West implements Direction {
		public int getId() { return 2; }
		public double front(Rectangle rect) { return -rect.x1; }
		public double back(Rectangle rect) { return -rect.x0; }
		public double minSide(Rectangle rect) { return rect.y0; }
		public double maxSide(Rectangle rect) { return rect.y1; }
		public double front(Rectangle2D rect) { return -rect.getMaxX(); }
		public double back(Rectangle2D rect) { return -rect.getMinX(); }
		public double minSide(Rectangle2D rect) { return rect.getMinY(); }
		public double maxSide(Rectangle2D rect) { return rect.getMaxY(); }
		public Direction turnLeft() { return SOUTH; }
		public Direction turnRight() { return NORTH; }		
		public RoutePencil createPencil(double front, double back, double min, double max, double penalty, RoutePencil parent) {
			return new RoutePencil(-back, min, -front, max, penalty, this, parent);
		}		
		public double getDir(double x, double y) { return -x; }	
		public double getSide(double x, double y) { return y; }
	}
	
	public static class North implements Direction {
		public int getId() { return 3; }
		public double front(Rectangle rect) { return -rect.y1; }
		public double back(Rectangle rect) { return -rect.y0; }
		public double minSide(Rectangle rect) { return rect.x0; }
		public double maxSide(Rectangle rect) { return rect.x1; }
		public double front(Rectangle2D rect) { return -rect.getMaxY(); }
		public double back(Rectangle2D rect) { return -rect.getMinY(); }
		public double minSide(Rectangle2D rect) { return rect.getMinX(); }
		public double maxSide(Rectangle2D rect) { return rect.getMaxX(); }
		public Direction turnLeft() { return WEST; }
		public Direction turnRight() { return EAST; }		
		public RoutePencil createPencil(double front, double back, double min, double max, double penalty, RoutePencil parent) {
			return new RoutePencil(min, -back, max, -front, penalty, this, parent);
		}		
		public double getDir(double x, double y) { return -y; }	
		public double getSide(double x, double y) { return x; }
	}
}
