/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.elementclass.canvas;

import java.awt.Color;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;

import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.SceneGraphNodeKey;
import org.simantics.g2d.element.handler.SceneGraph;
import org.simantics.scenegraph.Node;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.nodes.ShapeNode;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;
import org.simantics.utils.page.MarginUtils;

/**
 * Initializes the scene graph of another diagram inside the internal bounds of
 * the element.
 * 
 * <p>
 * Used keys for input element:
 * <ul>
 * <li>DiagramHints.KEY_DIAGRAM</li>
 * <li>DiagramPainter.KEY_VIEWPORT</li>
 * </ul>
 * 
 * <p>
 * Supports:
 * <ul>
 * <li>BorderColor</li>
 * <li>FillColor</li>
 * </ul>
 * 
 * @see MarginUtils for determining viewport or
 * @see ElementViewport lock viewport on an element
 * 
 * @author Toni Kalajainen
 */
public class DiagramPainter implements SceneGraph {

    public static final Key            KEY_VIEWPORT     = new KeyOf(AffineTransform.class, "DiagramPainter.VIEWPORT");

    private static final long          serialVersionUID = 6500535141363250713L;

    public static final DiagramPainter INSTANCE         = new DiagramPainter();

    private static final Color         FILM_COLOR       = new Color(0.0f, 0.0f, 0.0f, 0.3f);

    private static final Key           SG_NODE          = new SceneGraphNodeKey(Node.class, "SUB_SG_NODE");

    @Override
    public void cleanup(IElement e) {
        ElementUtils.removePossibleNode(e, SG_NODE);
    }

    @Override
    public void init(IElement e, G2DParentNode parent) {
        G2DParentNode node = (G2DParentNode) e.getHint(SG_NODE);
        if (node == null) {
            node = parent.addNode(G2DParentNode.class);
            e.setHint(SG_NODE, node);
        }

        IDiagram d = e.getHint(DiagramHints.KEY_DIAGRAM);
        Rectangle2D bounds = ElementUtils.getElementBounds(e);
        AffineTransform viewport = e.getHint(KEY_VIEWPORT);
        Color bc = ElementUtils.getBorderColor(e);
        Color fc = ElementUtils.getFillColor(e);

        node.removeNodes();

        if (fc != null) {
            ShapeNode fcn = node.addNode(ShapeNode.class);
            fcn.setColor(fc);
            fcn.setFill(true);
            fcn.setShape(bounds);
        }

        if (bc != null) {
            ShapeNode bcn = node.addNode(ShapeNode.class);
            Stroke stroke = e.getHint(ElementHints.KEY_STROKE);
            if (stroke != null)
                bcn.setStroke(stroke);
            bcn.setScaleStroke(true);
            bcn.setColor(bc);
            bcn.setShape(bounds);
        }

        // No diagram
        if (d==null) {
            ShapeNode shape = node.addNode(ShapeNode.class);
            ShapeNode line1 = node.addNode(ShapeNode.class);
            ShapeNode line2 = node.addNode(ShapeNode.class);

            shape.setColor(FILM_COLOR);
            shape.setFill(true);
            shape.setShape(bounds);

            line1.setColor(Color.WHITE);
            line1.setShape(new Line2D.Double(bounds.getMinX(), bounds.getMinX(), bounds.getMaxX(), bounds.getMaxY()));

            line2.setColor(Color.WHITE);
            line2.setShape(new Line2D.Double(bounds.getMaxX(), bounds.getMinX(), bounds.getMinX(), bounds.getMaxY()));
        } else {
            if (viewport==null)
                viewport = new AffineTransform();
// FIXME: we need to get elementpainter..
//            ElementPainter ep = ctx.getSingleItem(ElementPainter.class);
//            ep.paintDiagram(node, d, null);
        }
    }
}
