/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.element.handler.impl;

import java.awt.Shape;
import java.awt.geom.AffineTransform;

import org.simantics.g2d.diagram.handler.Topology.Terminal;
import org.simantics.g2d.utils.geom.DirectionSet;
import org.simantics.utils.Container;
import org.simantics.utils.DataContainer;

/**
 * An arbitrary object based implementation of the g2d {@link Terminal}
 * interface.
 * 
 * <p>
 * Elements that intend to contain terminal and use
 * <code>DiagramGraphSynchronizer</code> must use this implementation to
 * represent the terminals of the element.
 * </p>
 * 
 * <p>
 * Also contains a relative transformation with respect to its parent, a set of
 * allowed drawing directions and a graphical shape to depict the terminal.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public abstract class ObjectTerminal implements Terminal {

    private static final Container<Shape> NO_SHAPE = new DataContainer<Shape>(null);

    private final Object          data;
    private final AffineTransform transform;
    private final DirectionSet    ds;
    private final Container<Shape> shape;

    public ObjectTerminal(Object data, AffineTransform transform, DirectionSet ds, Container<Shape> shape) {
        assert data != null;
        this.data = data;
        this.transform = transform;
        this.ds = ds;
        this.shape = shape != null ? shape : NO_SHAPE;
    }

    public ObjectTerminal(Object data, AffineTransform transform, DirectionSet ds, Shape shape) {
        this(data, transform, ds, new DataContainer<Shape>(shape));
    }

    public ObjectTerminal(Object data, AffineTransform transform, DirectionSet ds) {
        this(data, transform, ds, NO_SHAPE);
    }

    public ObjectTerminal(Object data, AffineTransform transform) {
        this(data, transform, DirectionSet.ANY, NO_SHAPE);
    }

    public ObjectTerminal(Object data) {
        this(data, new AffineTransform(), DirectionSet.ANY, NO_SHAPE);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "[" + data + ", " + transform + ", " + shape + "]";
    }

    @Override
    public int hashCode() {
        return data.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null || getClass() != obj.getClass())
            return false;
        ObjectTerminal other = (ObjectTerminal) obj;
        return data.equals(other.data);
    }

    public Object getData() {
        return data;
    }

    public AffineTransform getTransform() {
        return transform;
    }

    public DirectionSet getDirections() {
        return ds;
    }

    public Shape getShape() {
        return shape.get();
    }

}