package org.simantics.export.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.osgi.service.prefs.BackingStoreException;
import org.osgi.service.prefs.Preferences;
import org.simantics.databoard.Accessors;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.accessor.RecordAccessor;
import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.accessor.reference.LabelReference;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.DatatypeConstructionException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.forms.DataboardForm;
import org.simantics.databoard.forms.DataboardForm.Problem;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.type.UnionType;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.intf.ContentType;
import org.simantics.export.core.intf.Exporter;
import org.simantics.export.core.intf.Format;
import org.simantics.export.core.intf.Publisher;
import org.simantics.export.core.manager.Content;
import org.simantics.export.core.manager.ExportAction;
import org.simantics.export.core.manager.ExportManager;
import org.simantics.export.core.manager.ExportPlan;
import org.simantics.export.core.manager.ExportWizardResult;
import org.simantics.export.core.util.ExporterUtils;
import org.simantics.utils.datastructures.Arrays;
import org.simantics.utils.datastructures.MapList;
import org.simantics.utils.datastructures.ToStringComparator;
import org.simantics.utils.datastructures.collections.CollectionUtils;
import org.simantics.utils.ui.ExceptionUtils;
import org.simantics.utils.ui.dialogs.ShowMessage;

/**
 * Dynamic Options page. Exporter, Importer and Format contributes options to this page. 
 *  
 * @author toni.kalajainen@semantum.fi
 */
public class OptionsPage extends WizardPage {

	public static String S_OUTPUT_OPTIONS = Messages.OptionsPage_OutputOptions;
	public static LabelReference P_OUTPUT_OPTIONS = new LabelReference( S_OUTPUT_OPTIONS );
	
	/** A reference to combo box selection */
	public static String S_PUBLISH = Messages.OptionsPage_PublishTo;
	public static ChildReference P_PUBLISH = ChildReference.compile(P_OUTPUT_OPTIONS, new LabelReference(S_PUBLISH));
	
	ExportContext ctx; 
	
	ScrolledComposite scroll;
	Composite composite;
	DataboardForm form;

	List<Content> selection;
	int selectionHash;
	String selectedPublisherId;
	
	public OptionsPage(ExportContext ctx) throws ExportException {
		super(Messages.OptionsPage_OptionsPage, Messages.OptionsPage_SelectExportOptions, null);
		
		this.ctx = ctx;
	}

	Listener modificationListener = new Listener() {		
		public void handleEvent(Event event) {
			if ( updatingForm ) return;
			validate();
		}
	};
	
	boolean updatingForm;
	Listener outputSettingsModifiedListener = new Listener() {		
		public void handleEvent(Event event) {
			updatingForm = true;
			try {
				Preferences contentScopePrefs = ctx.store.node( "Selection-"+selectionHash ); //$NON-NLS-1$
				Preferences workspaceScopePrefs = ctx.store;
				
				Composite outputOptions = (Composite) form.getControl(composite, P_OUTPUT_OPTIONS);
				Combo publishTo = (Combo) form.getControl(composite, P_PUBLISH);			
				String newPublisherLabel = publishTo.getText();
				Publisher newPublisher = ctx.eep.getPublisherByLabel(newPublisherLabel);
				String newPublisherId = newPublisher==null?null:newPublisher.id();
				//if ( newPublisherId.equals(selectedPublisherId) ) return;
				
				// Save Preferences
				RecordBinding binding = ctx.databoard.getMutableBinding( form.type() );
				Object obj = binding.createDefault();
				form.readFields(composite, binding, obj);
				Variant options = new Variant(binding, obj);
				
				Publisher oldPublisher = ctx.eep.getPublisher( selectedPublisherId );
				if ( oldPublisher!=null ) {
					ChildReference oldOptionsRef = new LabelReference( oldPublisher.label() );
					try {
						Variant locationOptions = options.getComponent( oldOptionsRef );
						oldPublisher.publisherClass().savePref(locationOptions, contentScopePrefs, workspaceScopePrefs);
					} catch ( AccessorConstructionException ee ) {}
				}

				List<Content> manifest = getManifestFor(selection, options);
				cleanPublisherFromGroup(selectedPublisherId);
				addPublisherToGroup(newPublisherId, manifest);
				
				outputOptions.pack(true);
				outputOptions.layout(true);
				composite.pack(true);
				composite.layout(true);
				
				RecordType dummy = new RecordType();
				RecordType newPublisherOptions = newPublisher.publisherClass().locationOptions(ctx, manifest);
				dummy.addComponent( newPublisherLabel, newPublisherOptions );
				binding = ctx.databoard.getMutableBinding( dummy );
				obj = binding.createDefault();
				options = new Variant(binding, obj);
				ChildReference locationOptionsRef = new LabelReference( newPublisherLabel );
				Variant locationOptions = options.getComponent( locationOptionsRef );
				newPublisher.publisherClass().fillDefaultPrefs(ctx, selection, options, locationOptions);
				newPublisher.publisherClass().loadPref(locationOptions, contentScopePrefs, workspaceScopePrefs);	
				//loadPublisherPref(newPublisherId, options, contentScopePrefs, workspaceScopePrefs);
				form.writeFields(outputOptions, (RecordBinding) options.getBinding(), options.getValue());
				
				selectedPublisherId = newPublisherId;

				updatingForm = false;
				validate();
			} catch ( BindingException e ) {
				setErrorMessage( e.getClass().getName()+": "+ e.getMessage() ); //$NON-NLS-1$
	            ExceptionUtils.logError(e);			
			} catch (AccessorConstructionException e) {
				setErrorMessage( e.getClass().getName()+": "+ e.getMessage() ); //$NON-NLS-1$
	            ExceptionUtils.logError(e);			
			} catch (AccessorException e) {
				setErrorMessage( e.getClass().getName()+": "+ e.getMessage() ); //$NON-NLS-1$
	            ExceptionUtils.logError(e);			
			} catch (ExportException e) {
				setErrorMessage( e.getClass().getName()+": "+ e.getMessage() ); //$NON-NLS-1$
	            ExceptionUtils.logError(e);			
			} finally {
				updatingForm = false;
			}
		}
	};
	
	@Override
	public void createControl(Composite parent) {
		
	    scroll = new ScrolledComposite(parent, SWT.V_SCROLL);
	    composite = new Composite(scroll, 0);
		composite.setLayout( new GridLayout(3, false) );			
		composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 3, 1));			
        scroll.setContent(composite);
        scroll.setExpandHorizontal(true);
        scroll.setExpandVertical(false);
        scroll.getVerticalBar().setIncrement(scroll.getVerticalBar().getIncrement()*8);

	    form = new DataboardForm();
	    form.setFirstColumnWidth(200);

		composite.pack();
		setControl(scroll);
        setPageComplete(true);
        
	}
	
	public void update(List<Content> contents) 
	{
		try {
			// 0. Initialization
			Set<String> contentUris = new TreeSet<String>();
			MapList<String, Content> uriToContentMap = new MapList<String, Content>();
			MapList<ContentType, String> contentByContentType = new MapList<ContentType, String>();
			MapList<Format, String> contentByFormat = new MapList<Format, String>();
			List<Format> formats = new ArrayList<Format>();
			Map<String, Format> contentFormatMap = new TreeMap<String, Format>( String.CASE_INSENSITIVE_ORDER );
			for (Content content : contents) {
				contentUris.add(content.url);
				ContentType ct = ctx.eep.getContentType( content.contentTypeId );
				contentByContentType.add(ct, content.url);
				Format format = ctx.eep.getFormat( content.formatId );
				contentByFormat.add(format, content.url);
				uriToContentMap.add(content.url, content);
			}
			formats.addAll( contentByFormat.getKeys() );
			
			Collections.sort(formats, new ToStringComparator());
						
			for (Content content : contents) {
				String id = content.formatId;
				Format format = contentFormatMap.get( id );
				if ( format != null ) continue;
				format = ctx.eep.getFormat(id);
				contentFormatMap.put(id, format);
			}

			MapList<Exporter, String> exporterContentMap = new MapList<Exporter, String>();
			TreeMap<String, Exporter> orderedExporters = new TreeMap<String, Exporter>( String.CASE_INSENSITIVE_ORDER );
			for (Content content : contents) {
				for (Exporter exporter : ctx.eep.getExporters(content.formatId, content.contentTypeId) ) {
					exporterContentMap.add(exporter, content.url);
					orderedExporters.put(exporter.formatId()+exporter.contentTypeId()+exporter.exportAction().getClass(), exporter);
				}
			}
			
			// 1. Save selections from previous form
			savePrefs();
			
			// 2. Clear previous form
			form.clear(composite);

			// 3. Create options record
			RecordType optionsRecord = new RecordType();

			// Add Output options box
		    RecordType outputOptions = new RecordType();
			
			for ( Format format : formats ) {
				if ( format.isGroupFormat() && contentByFormat.getValues(format).size()>1 ) {
					outputOptions.addComponent(NLS.bind(Messages.OptionsPage_Merge, format.fileext()), Datatypes.BOOLEAN); //$NON-NLS-2$
				}
				
				if ( format.isContainerFormat() ) {
					List<String> formatsContentUris = contentByFormat.getValues(format);
					int attachmentCount = 0;
					for ( String contentUri : formatsContentUris ) {
						for ( Content content : uriToContentMap.getValues(contentUri) ) {
							if ( !content.formatId.equals(format.id()) ) attachmentCount++;
						}
					}
					// Add as possible attachment, all contents that don't have this format
					// their their content type.
					for ( Content content : contents ) {
						if ( ctx.eep.getExporters(format.id(), content.contentTypeId).length == 0) attachmentCount++;
					}
					
					if ( attachmentCount > 0 ) { 
						outputOptions.addComponent(NLS.bind(Messages.OptionsPage_IncludeAttachmentTo, format.fileext()), Datatypes.BOOLEAN);
						outputOptions.addComponent(NLS.bind(Messages.OptionsPage_ExportAttachmentOf, format.fileext()), Datatypes.BOOLEAN); 
					}
				}
			}
			
			UnionType publisherType = new UnionType();
			for (Publisher publisher : ctx.eep.publishers()) publisherType.addComponent(publisher.label(), Datatypes.VOID);
			outputOptions.addComponent(S_PUBLISH, publisherType);
			
		    optionsRecord.addComponent(S_OUTPUT_OPTIONS, outputOptions);		    
		    
			// Add Format specific boxes
			for (Format format : contentFormatMap.values()) {
				RecordType formatOptions = format.formatActions().options(ctx);
				if ( formatOptions==null ) continue;
				optionsRecord.mergeRecord( formatOptions );
			}
	
			// Add Exporter specific boxes
			for (Exporter exporter : orderedExporters.values()) {				
				List<String> exportSpecificContents = exporterContentMap.getValues(exporter);
				if ( exportSpecificContents==null || exportSpecificContents.isEmpty() ) continue;
				RecordType exporterOptions = exporter.exportAction().options(ctx, exportSpecificContents);
				if ( exporterOptions == null ) continue;
				optionsRecord.mergeRecord( exporterOptions );
			}

			// 4. Load default and previous selections of the options ( All but publisher )
			ctx.databoard.clear();
			RecordBinding optionsBinding = ctx.databoard.getMutableBinding( optionsRecord );
			Object optionsObj = optionsBinding.createDefaultUnchecked();
			Variant options = new Variant(optionsBinding, optionsObj);
			{
				Preferences workspaceScopePrefs = ctx.store;
				Preferences contentScopePrefs = ctx.store( contents );
				
				for (Exporter exporter : orderedExporters.values()) {
					exporter.exportAction().fillDefaultPrefs(ctx, options);
					exporter.exportAction().loadPref(options, contentScopePrefs, workspaceScopePrefs);
				}
				
				for (Format format : contentByFormat.getKeys()) {
					format.formatActions().fillDefaultPrefs( ctx, options );
					format.formatActions().loadPref(options, contentScopePrefs, workspaceScopePrefs);
				}
				
				fillDefaultPrefs(options);
				loadPref(options, contentScopePrefs, workspaceScopePrefs);
			}
			
			// 5. Create form
			form.addFields(composite, optionsRecord);
			form.addListener(composite, form.type(), modificationListener);
			Composite outputOptionsGroup = (Composite) form.getControl(composite, P_OUTPUT_OPTIONS);
			RecordType dummy = new RecordType();
			dummy.addComponent(S_OUTPUT_OPTIONS, outputOptions);
			form.addListener(outputOptionsGroup, dummy, outputSettingsModifiedListener);
			form.writeFields(composite, optionsBinding, optionsObj);
			
			// 6. Add publisher
			{
				selection = contents;
				Preferences workspaceScopePrefs = ctx.store;
				Preferences contentScopePrefs = ctx.store.node( "Selection-"+selectionHash ); //$NON-NLS-1$
				selectedPublisherId = ctx.store.get("publisherId", "");		 //$NON-NLS-1$ //$NON-NLS-2$
				Publisher publisher = ctx.eep.getPublisher(selectedPublisherId);
				if ( publisher != null ) {
					
					// 6A. Manifest
					List<Content> manifest = getManifestFor(contents, options);
					
					// 6B. Default and previous settings
					String label = publisher.label();
					RecordType publisherOptionsType = publisher.publisherClass().locationOptions(ctx, manifest);
					RecordType publisherOptionsRootType = new RecordType();
					publisherOptionsRootType.addComponent(label, publisherOptionsType);
					RecordBinding publisherOptionsRootBinding = ctx.databoard.getMutableBinding( publisherOptionsRootType );
					Object publisherOptionsRootObj = publisherOptionsRootBinding.createDefaultUnchecked();
					Variant publisherOptionsRoot = new Variant(publisherOptionsRootBinding, publisherOptionsRootObj);
					try {
						ChildReference locationOptionsRef = new LabelReference( label );
						Variant locationOptions = publisherOptionsRoot.getComponent( locationOptionsRef );
						publisher.publisherClass().fillDefaultPrefs(ctx, selection, options, locationOptions);
						publisher.publisherClass().loadPref(locationOptions, contentScopePrefs, workspaceScopePrefs);	
					} catch ( AccessorConstructionException ee ) {
					}
					
					// 6C. Add Publisher form
					addPublisherToGroup(selectedPublisherId, manifest);
					form.writeFields(composite, publisherOptionsRootBinding, publisherOptionsRootObj);
				}
			}

			// 8. Validate page
			composite.pack();
			validate();
			
		} catch (BindingException e) {
            ExceptionUtils.logError(e);			
			ShowMessage.showError(Messages.OptionsPage_UnExpectedError, e.getClass().getName()+" "+e.getMessage()); //$NON-NLS-2$
			throw new RuntimeBindingException(e);
		} catch (ExportException e) {
            ExceptionUtils.logError(e);			
			ShowMessage.showError(Messages.OptionsPage_UnExpectedError, e.getClass().getName()+" "+e.getMessage()); //$NON-NLS-2$
		} catch (DatatypeConstructionException e) {
            ExceptionUtils.logError(e);			
			ShowMessage.showError(Messages.OptionsPage_UnExpectedError, e.getClass().getName()+" "+e.getMessage()); //$NON-NLS-2$
		} catch (AccessorConstructionException e) {
            ExceptionUtils.logError(e);			
			ShowMessage.showError(Messages.OptionsPage_UnExpectedError, e.getClass().getName()+" "+e.getMessage()); //$NON-NLS-2$
		} catch (AccessorException e) {
            ExceptionUtils.logError(e);			
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
	}

	/**
	 * Saves publisher prefs from the UI to preference nodes.
	 * 
	 * @throws ExportException
	 */
	void savePublisherPref(String publisherId, Variant options, Preferences contentScopePrefs, Preferences workspaceScopePrefs) throws ExportException {
		try {
//			Preferences contentScopePrefs = ctx.store.node( "Selection-"+selectionHash );
//			Preferences workspaceScopePrefs = ctx.store;

//			RecordBinding binding = ctx.databoard.getMutableBinding( form.type() );
//			Object obj = binding.createDefault();
//			form.readFields(composite, binding, obj);
//			Variant options = new Variant(binding, obj);
			
			Publisher publisher = ctx.eep.getPublisher( publisherId );
			if ( publisher==null ) return;
			Variant locationOptions = options.getComponent( new LabelReference(publisher.label()) );
			publisher.publisherClass().savePref(locationOptions, contentScopePrefs, workspaceScopePrefs);
//		} catch (BindingException e) {
//			throw new ExportException( e );
		} catch (AccessorConstructionException e) {
            ExceptionUtils.logError(e);			
			//throw new ExportException( e );
//		} catch (AccessorException e) {
//			throw new ExportException( e );
		}
	}
	
	void loadPublisherPref(String publisherId, Variant options, Preferences contentScopePrefs, Preferences workspaceScopePrefs) throws ExportException {
		try {
//			Preferences contentScopePrefs = ctx.store.node( "Selection-"+selectionHash );
//			Preferences workspaceScopePrefs = ctx.store;
			
			Publisher publisher = ctx.eep.getPublisher( publisherId );
			if ( publisher == null ) return;
			
			// There is a problem here if selection was built with manifest
//			RecordType optionsType = publisher.publiserClass().locationOptions(ctx, selection);			
//			RecordBinding binding = ctx.databoard.getMutableBinding( optionsType );
//			Object obj = binding.createDefault();
//			Variant locationOptions = new Variant(binding, obj);
			Variant locationOptions = options.getComponent( new LabelReference( publisher.label() ) );
			publisher.publisherClass().fillDefaultPrefs(ctx, selection, options, locationOptions);
			publisher.publisherClass().loadPref(locationOptions, contentScopePrefs, workspaceScopePrefs);	
//		} catch (BindingException e) {
//			throw new ExportException( e );
		} catch (AccessorConstructionException e) {
            ExceptionUtils.logError(e);			
			//throw new ExportException( e );
//		} catch (AccessorException e) {
//			throw new ExportException( e );
		}
	}

	/**
	 * Remove any publisher specific controls from output options.
	 */
	void cleanPublisherFromGroup(String oldPublisherId) {
		Publisher publisher = ctx.eep.getPublisher(oldPublisherId);
		if ( publisher != null ) {
			String fieldName = publisher.label();
			if ( form.type().hasComponent(fieldName)) {
				form.type().removeComponent(fieldName);
				ctx.databoard.clear();
			}
		}
		Combo publishTo = (Combo) form.getControl(composite, P_PUBLISH);
		Composite group = (Composite) publishTo.getParent();
		Control children[] = group.getChildren();
		int index = Arrays.indexOf( children, publishTo );
		for (int i=children.length-1; i>index; i--) {
			children[i].dispose();
		}
	}
	
	/**
	 * Add controls of a publisher id 
	 * 
	 * @param publisherId
	 * @throws ExportException 
	 */
	void addPublisherToGroup( String publisherId, List<Content> contents ) throws ExportException {
		try {
			Composite group = (Composite) form.getControl(composite, P_OUTPUT_OPTIONS);
			if ( group == null ) return;
			Publisher publisher = ctx.eep.getPublisher( publisherId );
			if ( publisher == null ) return;
			RecordType publisherOptions = publisher.publisherClass().locationOptions(ctx, contents);		
			//form.type().addComponent(publisher.label(), publisherOptions);
			RecordType options = new RecordType();
			options.addComponent(publisher.label(), publisherOptions);
			form.addFields(group, publisherOptions, publisher.label());
			form.addListener(group, options, modificationListener);
			ctx.databoard.clear();
		} catch (BindingException e) {
			throw new ExportException(e);
		} catch (AccessorConstructionException e) {
			throw new ExportException(e);
		} catch (AccessorException e) {
			throw new ExportException(e);
		}
	}
	
	List<Content> getManifestFor(List<Content> selection, Variant options) {		
		try {
			ExportWizardResult result = new ExportWizardResult();
			result.options = options;
			result.accessor = Accessors.getAccessor(result.options);
			result.contents = selection;
			result.type = (RecordType) options.type();
			result.publisherId = "file"; //$NON-NLS-1$
			
			List<ExportAction> actions = new ArrayList<ExportAction>();
			List<Content> manifest = new ArrayList<Content>(); 
			result.createExportActions(ctx, actions, manifest);
			return manifest;
		} catch (AccessorConstructionException e) {
			return selection;
		} catch (ExportException e) {
			e.printStackTrace();
			return selection;
		}
	}	
	
	public ExportWizardResult getOutput()
	throws ExportException
	{
		ExportWizardResult result = new ExportWizardResult();
		try {
			ctx.databoard.clear();
			result.type = form.type();
		    RecordBinding binding = (RecordBinding) ctx.databoard.getMutableBinding( result.type );
		    Object optionsObj = binding.createDefault();
		    result.options = new Variant(binding, optionsObj);
			form.readFields(composite, binding, optionsObj);
			result.accessor = Accessors.getAccessor(result.options);
			result.contents = selection;
			
			
			String publisherLabel = ExporterUtils.getUnionValue(result.accessor, P_PUBLISH);
			Publisher publisher = ctx.eep.getPublisherByLabel(publisherLabel);
			result.publisherId = publisher==null?"":publisher.id();			 //$NON-NLS-1$
			
		} catch (BindingException e) {
			throw new ExportException(e);			
		} catch (AccessorException e) {
			throw new ExportException(e);
		} catch (AccessorConstructionException e) {
			throw new ExportException(e);
		}
		
		return result;
	}
		
	void validate() {		
		List<Problem> problems = form.validate(composite);
		List<String> errs = DataboardForm.toStrings(problems);
		
		if ( errs.isEmpty() ) {
			try {
				ExportWizardResult result = getOutput();
				ExportPlan plan = new ExportPlan();
				result.createPlan(ctx, plan);
				ExportManager mgr = new ExportManager(result.options, ctx);
				errs.addAll( mgr.validate(ctx, plan) ) ;
			} catch (ExportException e) {
				errs.add(e.getMessage());
			}
		} else {
			CollectionUtils.unique( errs );			
		}
		

		setErrorMessage( errs.isEmpty() ? null : CollectionUtils.toString(errs, ", ") ); //$NON-NLS-1$
		setPageComplete( errs.isEmpty() );
	}
		
	/**
	 * Save wizard preferences
	 * 
	 * @param options
	 * @param contentScopePrefs
	 * @param workspaceScopePrefs
	 * @throws ExportException
	 */
	void savePref(Variant options, Preferences contentScopePrefs, Preferences workspaceScopePrefs) throws ExportException
	{
        try {
			RecordAccessor ra = Accessors.getAccessor(options);
			String publisherId = ExporterUtils.getUnionValue(ra, P_PUBLISH); 
			Publisher publisher = ctx.eep.getPublisherByLabel(publisherId);
			if ( publisher!=null ) workspaceScopePrefs.put("publisherId", publisher.id()); //$NON-NLS-1$
			
			if ( ra.type().hasComponent(P_OUTPUT_OPTIONS.label) ) {
				RecordAccessor rao = ra.getComponent( P_OUTPUT_OPTIONS );
								
				Pattern merge_pattern = Pattern.compile("Merge ([^\\s]*) content into one file"); //$NON-NLS-1$
				Pattern include_pattern = Pattern.compile("Include attachments to ([^\\s]*)"); //$NON-NLS-1$
				Pattern export_pattern = Pattern.compile("Export attachments of ([^\\s]*) to separate files"); //$NON-NLS-1$
				
				for (int i=0; i<rao.count(); i++) {
					String name = rao.type().getComponent(i).name;
					Matcher m;
					
					m = merge_pattern.matcher(name);
					if ( m.matches() ) {
						String fileExt = m.group(1);
						Format format = ctx.eep.getFormatByExt(fileExt);
						Boolean value = (Boolean) rao.getFieldValue(i, Bindings.BOOLEAN);
						String key = format.id()+"_merge"; //$NON-NLS-1$
						contentScopePrefs.putBoolean(key, value);
						workspaceScopePrefs.putBoolean(key, value);
					}
					
					m = include_pattern.matcher(name);
					if ( m.matches() ) {
						String fileExt = m.group(1);
						Format format = ctx.eep.getFormatByExt(fileExt);
						Boolean value = (Boolean) rao.getFieldValue(i, Bindings.BOOLEAN);
						String key = format.id()+"_include_attachments"; //$NON-NLS-1$
						contentScopePrefs.putBoolean(key, value);
						workspaceScopePrefs.putBoolean(key, value);
					}
					
					m = export_pattern.matcher(name);
					if ( m.matches() ) {
						String fileExt = m.group(1);
						Format format = ctx.eep.getFormatByExt(fileExt);
						Boolean value = (Boolean) rao.getFieldValue(i, Bindings.BOOLEAN);
						String key = format.id()+"_export_attachments"; //$NON-NLS-1$
						contentScopePrefs.putBoolean(key, value);
						workspaceScopePrefs.putBoolean(key, value);
					}
				}
			}
			
		} catch (AccessorConstructionException e) {
			throw new ExportException(e);
		} catch (AccessorException e) {
			throw new ExportException(e);
		}
		
	}

	/**
	 * Load wizard preferences
	 * 
	 * @param binding
	 * @param options
	 * @param contentScopePrefs
	 * @param workspaceScopePrefs
	 * @throws ExportException
	 */
	void loadPref(Variant options, Preferences contentScopePrefs, Preferences workspaceScopePrefs) throws ExportException
	{
		//Preferences workspaceScopePrefs = ctx.store;
        try {
			RecordAccessor ra = Accessors.getAccessor(options);		
			String publisherId = workspaceScopePrefs.get("publisherId", ""); //$NON-NLS-1$ //$NON-NLS-2$
			Publisher publisher = ctx.eep.getPublisher(publisherId);
			if ( publisher != null ) ExporterUtils.setUnionValue(ra, P_PUBLISH, publisher.label());
			
			if ( ra.type().hasComponent(P_OUTPUT_OPTIONS.label) ) {
				RecordAccessor rao = ra.getComponent( P_OUTPUT_OPTIONS );
								
				for (Format format : ctx.eep.formats()) {
					if ( format.isGroupFormat() ) {
						String key = format.id()+"_merge"; //$NON-NLS-1$
						Boolean value = null;
						if ( containsKey(contentScopePrefs, key) ) {
							value = contentScopePrefs.getBoolean(key, false); 
						} else if ( containsKey(workspaceScopePrefs, key) ) {
							value = workspaceScopePrefs.getBoolean(key, false);
						}
											
						if ( value != null ) {
							String key2 = "Merge "+format.fileext()+" content into one file"; //$NON-NLS-1$ //$NON-NLS-2$
							int index = rao.type().getComponentIndex2(key2);
							if ( index>=0 ) {
								rao.setFieldValue(index, Bindings.BOOLEAN, value);
							}
						}					
					}
					
					if ( format.isContainerFormat() ) {
						String key = format.id()+"_include_attachments"; //$NON-NLS-1$
						Boolean value = null;
						if ( containsKey(contentScopePrefs, key) ) {
							value = contentScopePrefs.getBoolean(key, false); 
						} else if ( containsKey(workspaceScopePrefs, key) ) {
							value = workspaceScopePrefs.getBoolean(key, false);
						}
											
						if ( value != null ) {
							String key2 = "Include attachments to "+format.fileext(); //$NON-NLS-1$
							int index = rao.type().getComponentIndex2(key2);
							if ( index>=0 ) {
								rao.setFieldValue(index, Bindings.BOOLEAN, value);
							}
						}					
					}
					
					if ( format.isContainerFormat() ) {
						String key = format.id()+"_export_attachments"; //$NON-NLS-1$
						Boolean value = null;
						if ( containsKey(contentScopePrefs, key) ) {
							value = contentScopePrefs.getBoolean(key, false); 
						} else if ( containsKey(workspaceScopePrefs, key) ) {
							value = workspaceScopePrefs.getBoolean(key, false);
						}
											
						if ( value != null ) {
							String key2 = "Export attachments of"+format.fileext()+" to separate files"; //$NON-NLS-2$
							int index = rao.type().getComponentIndex2(key2);
							if ( index>=0 ) {
								rao.setFieldValue(index, Bindings.BOOLEAN, value);
							}
						}					
					}
					
				}
			}
					
		} catch (AccessorConstructionException e) {
			throw new ExportException(e);
		} catch (AccessorException e) {
			throw new ExportException(e);
		}		
	}
	
	/**
	 * Save selections from previous form
	 */
	public void savePrefs() throws ExportException {
		try {
			int oldSelectionHash = selectionHash;			
			Preferences contentScopePrefs = ctx.store.node( "Selection-"+oldSelectionHash ); //$NON-NLS-1$
			Preferences workspaceScopePrefs = ctx.store;
					
			RecordBinding binding = ctx.databoard.getMutableBinding( form.type() );
			Object obj = binding.createDefault();
			Variant options = new Variant(binding, obj);
			form.readFields(composite, binding, obj);
						
			Combo publishTo = (Combo) form.getControl(composite, P_PUBLISH);		
			String publisherLabel = publishTo==null?null:publishTo.getText();
			if ( publisherLabel != null ) {
				Publisher publisher = ctx.eep.getPublisherByLabel( publisherLabel );
				if ( publisher!=null ) {
					try {
						ChildReference locationOptionsRef = new LabelReference(publisher.label());
						Variant locationOptions = options.getComponent( locationOptionsRef );
						publisher.publisherClass().savePref(locationOptions, contentScopePrefs, workspaceScopePrefs);
					} catch ( AccessorConstructionException e ) {} 
				}
			}
			
			for (Exporter exporter : ctx.eep.exporters()) {
				exporter.exportAction().savePref(options, contentScopePrefs, workspaceScopePrefs);
			}
					
			for (Format format : ctx.eep.formats()) {
				format.formatActions().savePref(options, contentScopePrefs, workspaceScopePrefs);
			}				
					
			savePref(options, contentScopePrefs, workspaceScopePrefs);
		} catch (BindingException e) {
			throw new ExportException( e ); 
		} catch (AccessorConstructionException e) {
			throw new ExportException( e ); 
		} catch (AccessorException e) {
			throw new ExportException( e ); 
		} catch (ExportException e) {
			throw new ExportException( e ); 
		}
	}

	public void fillDefaultPrefs(Variant options) throws ExportException {
        try {
			RecordAccessor ra = Accessors.getAccessor(options);

			if ( ra.type().hasComponent(P_OUTPUT_OPTIONS.label) ) {
				RecordAccessor rao = ra.getComponent( P_OUTPUT_OPTIONS );
				
				for (Format format : ctx.eep.formats()) {
					if ( format.isContainerFormat() ) {
						String key = "Include attachments to "+format.fileext(); //$NON-NLS-1$
						int index = rao.type().getComponentIndex2(key);
						if ( index>=0 ) rao.setFieldValue(index, Bindings.BOOLEAN, true);
						
						key = "Export attachments of "+format.fileext()+" to separate files";; //$NON-NLS-1$ //$NON-NLS-2$
						index = rao.type().getComponentIndex2(key);
						if ( index>=0 ) rao.setFieldValue(index, Bindings.BOOLEAN, true);
					}			
				}
				
			}
        	
		} catch (AccessorConstructionException e) {
			throw new ExportException(e);
		} catch (AccessorException e) {
			throw new ExportException(e);
		}
	}	
	
	static boolean containsKey(Preferences pref, String key) {
		try {
			for (String x : pref.keys()) if ( x.equals(key) ) return true;
		} catch (BackingStoreException e) {
			e.printStackTrace();
		} 
		return false;		
	}
		
}
