/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.content.EdgeResource;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.modeling.ModelingResources;

/**
 * An element assortment is used to categorize diagram contents in diagram
 * cut-copy-paste operations.
 * 
 * <p>
 * This version of {@link ElementAssortment} contains only the back-end objects
 * (see {@link ElementHints#KEY_OBJECT}) of IElement instances instead of the
 * IElement instances themselves. This version doesn't have the dependency on
 * the diagram runtime model that {@link ElementAssortment} has. This object can
 * be used even if the diagram runtime model and its elements are disposed.
 * 
 * @author Tuukka Lehtonen
 */
public class ElementObjectAssortment implements IElementAssortment {

    private EnumSet<ElementType>   contents;

    public Set<Object>             all;

    public final List<Resource>    nodeList       = new ArrayList<Resource>();
    public final Set<Resource>     nodes          = new HashSet<Resource>();
    public final Set<Resource>     connections    = new HashSet<Resource>();
    public final Set<EdgeResource> edges          = new HashSet<EdgeResource>();
    public final Set<Resource>     branchPoints   = new HashSet<Resource>();
    public final Set<Resource>     flags          = new HashSet<Resource>();
    public final Set<Resource>     references     = new HashSet<Resource>();
    public final Set<Resource>     monitors       = new HashSet<Resource>();
    public final Set<Resource>     others         = new HashSet<Resource>();
    public final Set<Object>       noncopyables   = new HashSet<Object>();

    public ElementObjectAssortment(ReadGraph graph, Collection<Resource> elements) throws DatabaseException {
        all = new HashSet<Object>(elements); 
        contents = EnumSet.noneOf(ElementType.class);
        sort(graph, elements);
    }

    private void sort(ReadGraph graph, Collection<Resource> elements) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);
        for (Resource element : elements) {
            if (graph.isInstanceOf(element, DIA.Flag)) {
                flags.add(element);
                contents.add(ElementType.Flag);
            } else if (graph.isInstanceOf(element, DIA.Connection)) {
                connections.add(element);
                contents.add(ElementType.Connection);
            } else if (graph.isInstanceOf(element, DIA.Monitor)) {
                monitors.add(element);
                contents.add(ElementType.Monitor);
            } else if (graph.isInstanceOf(element, MOD.ReferenceElement)) {
                references.add(element);
                contents.add(ElementType.Reference);
            } else if (graph.isInstanceOf(element, DIA.DefinedElement)) {
                nodeList.add(element);
                nodes.add(element);
                contents.add(ElementType.Node);
            } else if (graph.isInstanceOf(element, DIA.Element)) {
                others.add(element);
                contents.add(ElementType.Other);
            } 
        }
    }

	/**
     * @param set all the elements to initially sort out. This object assumes
     *        ownership of the set.
     */
    public static ElementObjectAssortment fromElements(Set<IElement> set) {
        return new ElementObjectAssortment( new ElementAssortment( set ) );
    }

    /**
     * @param set all the elements to initially sort out. This object assumes
     *        ownership of the set.
     */
    public ElementObjectAssortment(ElementAssortment assortment) {
        if (assortment == null)
            throw new IllegalArgumentException("null element assortment");
        load(assortment);
    }

    private void load(ElementAssortment es) {
        this.contents = es.contents;
        this.all = getElementObjectSet(es.all, Object.class);
        getElementObjects(es.nodeList, Resource.class, this.nodeList);
        getElementObjects(es.nodes, Resource.class, this.nodes);
        getElementObjects(es.connections, Resource.class, this.connections);
        getElementObjects(es.connections, Resource.class, this.connections);
        getElementObjects(es.edges, EdgeResource.class, this.edges);
        getElementObjects(es.branchPoints, Resource.class, this.branchPoints);
        getElementObjects(es.flags, Resource.class, this.flags);
        getElementObjects(es.references, Resource.class, this.references);
        getElementObjects(es.monitors, Resource.class, this.monitors);
        getElementObjects(es.others, Resource.class, this.others);
        getElementObjects(es.noncopyables, Object.class, this.noncopyables);
    }

    public Set<Object> getAll() {
        return new HashSet<Object>(all);
    }

    public boolean contains(ElementType et) {
        return contents.contains(et);
    }

    public boolean contains(Collection<ElementType> ets) {
        return contents.containsAll(ets);
    }

    public boolean containsAny(Collection<ElementType> ets) {
        for (ElementType et : ets)
            if (contents.contains(et))
                return true;
        return false;
    }

    @Override
    public int count(ElementType et) {
        switch (et) {
            case BranchPoint: return branchPoints.size();
            case Connection: return connections.size();
            case Edge: return edges.size();
            case Flag: return flags.size();
            case Reference: return references.size();
            case Monitor: return monitors.size();
            case Node: return nodes.size();
            case Other: return others.size();
            case NonCopyable: return noncopyables.size();
            default: return 0;
        }
    }

    @Override
    public String toString() {
        StringBuilder b = new StringBuilder();
        b.append("ElementObjectAssortment:\n");
        b.append("\t CONTAINS: ");
        b.append(contents);
        b.append("\n");
        for(Resource e : nodes) b.append("\t-node " + e + "\n");
        for(Resource e : connections) b.append("\t-connection " + e + "\n");
        for(EdgeResource e : edges) b.append("\t-edge " + e + "\n");
        for(Resource e : branchPoints) b.append("\t-branch " + e + "\n");
        for(Resource e : flags) b.append("\t-flag " + e + "\n");
        for(Resource e : references) b.append("\t-reference " + e + "\n");
        for(Resource e : monitors) b.append("\t-monitor " + e + "\n");
        for(Object e : others) b.append("\t-other " + e + "\n");
        for(Object e : noncopyables) b.append("\t-non-copyable " + e + "\n");
        return b.toString();
    }

    public boolean isEmpty() {
        return all.isEmpty();
    }

    @SuppressWarnings("unchecked")
    public static <T> Collection<T> getElementObjects(Collection<IElement> elements, Class<T> clazz, Collection<T> result) {
        for (IElement e : elements) {
            Object o = ElementUtils.getObject(e);
            if (clazz.isInstance(o))
                result.add((T) o);
        }
        return result;
    }

    public static <T> Set<T> getElementObjectSet(Collection<IElement> elements, Class<T> clazz) {
        Set<T> result = new HashSet<T>();
        getElementObjects(elements, clazz, result);
        return result;
    }

}
