/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.content;

import gnu.trove.map.hash.THashMap;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.simantics.db.Resource;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramContentChanges {

    public final Map<Resource, Change>        elements;
    public final Map<Resource, Change>        nodes;
    public final Map<Resource, Change>        connections;
    public final Map<EdgeResource, Change>    connectionSegments;
    public final Map<Resource, Change>        branchPoints;

    public final Map<Resource, Change>        routeGraphConnections;
    public final Map<EdgeResource, Change>    routeLinks;
    public final Map<Resource, Change>        routeLines;
    public final Map<Resource, Change>        routePoints;

    public boolean                            elementOrderChanged = false;

    public static final DiagramContentChanges EMPTY;

    static {
        EMPTY = new DiagramContentChanges(true);
    }

    private DiagramContentChanges(boolean empty) {
        this.elements = Collections.emptyMap();
        this.nodes = Collections.emptyMap();
        this.connections = Collections.emptyMap();
        this.connectionSegments = Collections.emptyMap();
        this.branchPoints = Collections.emptyMap();
        this.routeGraphConnections = Collections.emptyMap();
        this.routeLinks = Collections.emptyMap();
        this.routeLines = Collections.emptyMap();
        this.routePoints = Collections.emptyMap();
    }

    public DiagramContentChanges() {
        this.elements = new THashMap<Resource, Change>();
        this.nodes = new THashMap<Resource, Change>();
        this.connections = new THashMap<Resource, Change>();
        this.connectionSegments = new THashMap<EdgeResource, Change>();
        this.branchPoints = new THashMap<Resource, Change>();
        this.routeGraphConnections = new THashMap<Resource, Change>();
        this.routeLinks = new THashMap<EdgeResource, Change>();
        this.routeLines = new THashMap<Resource, Change>();
        this.routePoints = new THashMap<Resource, Change>();
    }

    public void markElementOrderChanged() {
        elementOrderChanged = true;
    }

    public boolean isEmpty() {
        return elements.isEmpty()
        && nodes.isEmpty()
        && connections.isEmpty()
        && connectionSegments.isEmpty()
        && branchPoints.isEmpty()
        && routeGraphConnections.isEmpty()
        && routeLinks.isEmpty()
        && routeLines.isEmpty()
        && routePoints.isEmpty()
        && !elementOrderChanged;
    }

    public <T> Set<T> pick(Map<T, Change> map, Change change) {
        Set<T> result = new HashSet<T>();
        for (Map.Entry<T, Change> entry : map.entrySet()) {
            if (entry.getValue() == change)
                result.add(entry.getKey());
        }
        return result;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[elements=");
        sb.append(toString(count(elements)));
        sb.append(", nodes=");
        sb.append(toString(count(nodes)));
        sb.append(", connection=");
        sb.append(toString(count(connections)));
        sb.append(", connection segments=");
        sb.append(toString(count(connectionSegments)));
        sb.append(", branch points=");
        sb.append(toString(count(branchPoints)));
        sb.append(", routegraph connections=");
        sb.append(toString(count(routeGraphConnections)));
        sb.append(", route links=");
        sb.append(toString(count(routeLinks)));
        sb.append(", route lines=");
        sb.append(toString(count(routeLines)));
        sb.append(", route points=");
        sb.append(toString(count(routePoints)));
        sb.append(", element order changed=");
        sb.append(elementOrderChanged);
        sb.append("]");
        return sb.toString();
    }

    private static String toString(int[] changes) {
        return
        "[" + Change.ADDED.toString() + "=" + changes[Change.ADDED.ordinal()] +
        ", " + Change.REMOVED.toString() + "=" + changes[Change.REMOVED.ordinal()] +
        //", " + Change.POSSIBLY_MODIFIED.toString() + "=" + changes[Change.POSSIBLY_MODIFIED.ordinal()] +
        "]";
    }

    private static int[] count(Map<?, Change> map) {
        int[] result = new int[Change.values().length];
        for (Change change : map.values()) {
            result[change.ordinal()]++;
        }
        return result;
    }

}