/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.content;

import java.util.HashSet;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceAsyncRead;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.ErrorHandler;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class ConnectionPartRequest extends ResourceAsyncRead<ConnectionPartData> {

    final ErrorHandler    errorHandler;

    final DiagramResource dr;

    public ConnectionPartRequest(ErrorHandler errorHandler, DiagramResource dr, Resource resource) {
        super(resource);
        this.errorHandler = errorHandler;
        this.dr = dr;
    }

    @Override
    public void perform(AsyncReadGraph graph, final AsyncProcedure<ConnectionPartData> procedure) {

        final AtomicInteger ready = new AtomicInteger(3);
        final ConnectionPartData result = new ConnectionPartData();
        final HashSet<EdgeResource> connectorEdges = new HashSet<EdgeResource>();
        final HashSet<EdgeResource> branchEdges = new HashSet<EdgeResource>();

        result.edges = connectorEdges;
        result.branchPoints = new ConcurrentLinkedQueue<Resource>();

        AsyncMultiProcedure<Resource> locationProcedure = new AsyncMultiProcedure<Resource>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                errorHandler.error("forEachObject(connection, Has I/O) failure", throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, final Resource location) {

                ready.incrementAndGet();

                graph.forEachObject(location, dr.AreConnected, new AsyncMultiProcedure<Resource>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        errorHandler.error("forEachObject(:I/O Connector, Are Connected) failure", throwable);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, Resource connectedTo) {
                        synchronized (connectorEdges) {
                            connectorEdges.add(new EdgeResource(location, connectedTo));
                        }
                    }

                    @Override
                    public void finished(AsyncReadGraph graph) {
                        if (ready.decrementAndGet() == 0) {
                            connectorEdges.addAll(branchEdges);
                            procedure.execute(graph, result);
                        }
                    }

                });

            }

            @Override
            public void finished(AsyncReadGraph graph) {
                if (ready.decrementAndGet() == 0) {
                    connectorEdges.addAll(branchEdges);
                    procedure.execute(graph, result);
                }
            }

        };

        AsyncMultiProcedure<Resource> locationProcedure2 = new AsyncMultiProcedure<Resource>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                errorHandler.error("forEachObject(connection, Has Branch Point) failure", throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, final Resource location) {

                ready.incrementAndGet();

                result.branchPoints.add(location);

                graph.forEachObject(location, dr.AreConnected, new AsyncMultiProcedure<Resource>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        errorHandler.error("forEachObject(:Branch Point, Are Connected) failure", throwable);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, Resource connectedTo) {
                        synchronized (branchEdges) {
                            branchEdges.add(new EdgeResource(location, connectedTo));
                        }
                    }

                    @Override
                    public void finished(AsyncReadGraph graph) {
                        if (ready.decrementAndGet() == 0) {
                            connectorEdges.addAll(branchEdges);
                            procedure.execute(graph, result);
                        }
                    }

                });

            }

            @Override
            public void finished(AsyncReadGraph graph) {
                if (ready.decrementAndGet() == 0) {
                    connectorEdges.addAll(branchEdges);
                    procedure.execute(graph, result);
                }
            }

        };

        AsyncMultiProcedure<Resource> locationProcedure3 = new AsyncMultiProcedure<Resource>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                errorHandler.error("forEachObject(connection, Has I/O) failure", throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, final Resource location) {

                ready.incrementAndGet();

                graph.forEachObject(location, dr.AreConnected, new AsyncMultiProcedure<Resource>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        errorHandler.error("forEachObject(:I/O Connector, Are Connected) failure", throwable);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, Resource connectedTo) {
                        synchronized (connectorEdges) {
                            connectorEdges.add(new EdgeResource(connectedTo, location));
                        }
                    }

                    @Override
                    public void finished(AsyncReadGraph graph) {
                        if (ready.decrementAndGet() == 0) {
                            connectorEdges.addAll(branchEdges);
                            procedure.execute(graph, result);
                        }
                    }

                });

            }

            @Override
            public void finished(AsyncReadGraph graph) {
                if (ready.decrementAndGet() == 0) {
                    connectorEdges.addAll(branchEdges);
                    procedure.execute(graph, result);
                }
            }

        };

        graph.forEachObject(resource, dr.HasPlainConnector, locationProcedure);
        graph.forEachObject(resource, dr.HasBranchPoint, locationProcedure2);
        graph.forEachObject(resource, dr.HasArrowConnector, locationProcedure3);

    }

}
