/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db;

import org.simantics.databoard.accessor.Accessor;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.binary.RandomAccessBinary;
import org.simantics.db.exception.BindingException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.request.AsyncMultiRead;

/**
 * For accessing and manipulating the graph data.
 *
 * Instantiated by Session for use only during a transaction.
 * 
 * @author Antti Villberg
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi> -docs
 * @see AsyncMultiRead
 * @see Resource
 * @see Statement
 */
public interface WriteGraph extends WriteOnlyGraph, ReadGraph {

    /**
     * Makes sure that the statements (s,p,o) and (o,p',s) are found in the
     * graph, where p' is the inverse predicate of p. Contrary to
     * {@link WriteOnlyGraph#claim(Resource, Resource, Resource, Resource)} this
     * method assures that the the statement and its inverse are semantically
     * valid after the invocation of this method.
     * 
     * @param subject subject, i.e. source resource of the statement to be
     *        claimed
     * @param predicate predicate resource of the statement to be claimed
     * @param object object, i.e. target resource of the statement to be claimed
     * @throws ServiceException
     */
    void claim(Resource subject, Resource predicate, Resource object) throws ServiceException;

    /**
     * Sets literal value related to the specified resource with the specified
     * predicate. If such value exists (s,p), the value is overridden with the
     * new specified value.
     * 
     * @param resource
     * @param predicate
     * @param value Value of the literal (boxed primitive/String or
     *        primitive/String array)
     * @throws ManyObjectsForFunctionalRelationException
     */

    /**
     * Claims
     * -(resource, predicate, ?o) and when applicable (?o, inverse of predicate, resource)
     * -(?o, L0.InstanceOf, L0.Literal or type) (user type is used if provided)
     * -(?o, L0.HasDatatype, <data type of value>) (is assumed to be asserted if type is given)
     * -(?o contains value) 
     * 
     * For L0 literals type is inferred if not provided by the user. Else if type is not provided L0.Literal is used.
     * If the user does not provide binding Bindings.getBinding(value.getClass()) is consulted.
     * Assumes that predicate is functional.
     * 
     * @throws ServiceException
     */
    void claimLiteral(Resource resource, Resource predicate, Object value) throws ManyObjectsForFunctionalRelationException, ServiceException;
    void claimLiteral(Resource resource, Resource predicate, Object value, Binding binding) throws BindingException, ManyObjectsForFunctionalRelationException, ServiceException;
    void claimLiteral(Resource resource, Resource predicate, Resource type, Object value) throws BindingException, ManyObjectsForFunctionalRelationException, ServiceException;
    void claimLiteral(Resource resource, Resource predicate, Resource type, Object value, Binding binding) throws BindingException, ManyObjectsForFunctionalRelationException, ServiceException;
    void claimLiteral(Resource resource, Resource predicate, Resource inverse, Resource type, Object value) throws BindingException, ManyObjectsForFunctionalRelationException, ServiceException;
    void claimLiteral(Resource resource, Resource predicate, Resource inverse, Resource type, Object value, Binding binding) throws BindingException, ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * Returns an accessory that can be used to partially read or write the the
     * literal. The returned accessory is closed automatically when the
     * transaction ends. The modifications are written back to graph at
     * transaction completion.
     * 
     * @param <T>
     * @param resource is the parent of literal.
     * @param predicate is the relation for the literal.
     * @param datatype is the data type of the literal.
     * @param initialValue for the literal. If null then default value is used.
     * @return
     * @throws DatabaseException
     */
    <T extends Accessor> T newLiteral(Resource resource, Resource predicate, Datatype datatype, Object initialValue) throws DatabaseException;
    
    /**
     * Returns RandomAccessBinary that can be used to partially read or write the the
     * literal. The returned RandomAccessBinary is closed automatically when the
     * transaction ends. The modifications are written back to graph at
     * transaction completion.
     * 
     * @param <T>
     * @param resource is the parent of literal.
     * @param predicate is the relation for the literal.
     * @param datatype is the data type of the literal.
     * @param initialValue for the literal. If null then default value is used.
     * @return RandomAccessBinary to access the literal.
     * @throws DatabaseException
     */
    RandomAccessBinary createRandomAccessBinary(Resource resource, Resource predicate, Datatype datatype, Object initialValue)
    		throws DatabaseException;

    RandomAccessBinary createRandomAccessBinary(Resource resource, Datatype datatype, Object initialValue)
    		throws DatabaseException;
    
    /**
     * Makes sure that no statements matching the patterns (s,?p,?o) and
     * (?o,?p',s), where ?p' is the inverse predicate of ?p, exist in the graph.
     * In other words, removes all statements outgoing from the specified
     * resource along with the inverses of those statements.
     * 
     * @param subject
     * @throws ServiceException
     */
    void deny(Resource subject) throws ServiceException;

    /**
     * Makes sure that no statements matching the patterns (s,p,?o) and
     * (?o,p',s), where p' is the inverse predicate of p, exist in the graph.
     * Also statements where <code>isSubrelationOf(p, predicate)</code> returns
     * <code>true</code> shall be removed. In other words, removes all
     * statements outgoing from the specified resource with the specified
     * predicate or any of its subrelations, along with the inverses of those
     * statements.
     * 
     * @param subject
     * @throws ServiceException
     */
    void deny(Resource subject, Resource predicate) throws ServiceException;

    /**
     * Makes sure that no statements matching the patterns (s,p,o) and (o,p',s),
     * where p' is the inverse predicate of p, exist in the graph. Contrary to
     * {@link #denyStatement(Resource, Resource, Resource)}, all statements
     * where <code>isSubrelationOf(p, predicate)</code> returns
     * <code>true</code> shall be removed. In other words, removes all
     * statements between the specified subject and object with the specified
     * predicate or any of its subrelations, along with the inverses of those
     * statements.
     * 
     * @param subject
     * @param predicate
     * @param object
     * @throws ServiceException
     */
    void deny(Resource subject, Resource predicate, Resource object) throws ServiceException;

    void deny(Resource subject, Resource predicate, Resource inverse, Resource object) throws ServiceException;

    /**
     * Makes sure that no statements matching the patterns (s,p,o) and (o,p',s),
     * where p' is the inverse predicate of p, exist in the graph. In other
     * words, removes the specified statement and its possible inverse.
     * 
     * <p>
     * This method behaves exactly like {@link #deny(Statement)}, it just takes
     * the arguments as resources instead of a statement.
     * 
     * @param subject
     * @throws ServiceException
     * 
     * @see {@link #deny(Statement)}
     */
    void denyStatement(Resource subject, Resource predicate, Resource object) throws ServiceException;

    /**
     * Makes sure that the specified statements (s,p,o) and its inverse
     * statements (o,p',s), where p' is the inverse predicate of p, do not exist
     * in the graph.
     * 
     * <p>
     * This method behaves exactly like
     * {@link #denyStatement(Resource, Resource, Resource)}, it just takes the
     * arguments as a statement instead of 3 resources.
     * 
     * @param statement
     * 
     * @see #denyStatement(Resource, Resource, Resource)
     */
    void deny(Statement statement) throws ServiceException;

    /**
     * Removes all statements (resource,predicate,?o) and literal contained by
     * ?o.
     * 
     * @param resource
     * @param predicate
     * @throws ManyObjectsForFunctionalRelationException
     */
    void denyValue(Resource resource, Resource predicate) throws ManyObjectsForFunctionalRelationException, ServiceException;

}
