/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.variable;

import java.util.Collection;
import java.util.Set;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.NonWritableVariableException;
import org.simantics.db.layer0.request.PropertyInfo;
import org.simantics.db.layer0.variable.RVI.RVIPart;
import org.simantics.db.layer0.variable.Variables.Role;

/*
 * Variable is a standard i/o data interface to a model in Simantics. The variable interface provides
 *
 *  - Unified data addressing in string format (URI)
 *  - Flattening of reusable component hierarchies
 *  - Runtime values 
 *  - Derived values
 *
 * The Variable interface spans a tree of variables such that each node can have children and properties.
 *  - children are denoted by the '/' separator
 *  - properties are denoted by the '#' separator
 *
 * There are three kinds of URIs
 * 
 *  Variable URI:
 *   A standard random access identifier to a variable.
 *   Examples:  
 *    - http://www.asd.org/Projects/AprosModel1/BaseRealization/Diagram/PI_X#PI_MASS_FLOW
 *    - http://www.asd.org/Projects/AprosModel1/Experiment/UUID/Diagram/PI_X#PI_MASS_FLOW#Unit
 *  Realization URI:
 *   A random access identifier to a context under which a valuation of a model is presented.
 *   A realization URI is guaranteed to have a unique resource obtainable by ReadGraph.getResource()
 *   Examples:
 *    - http://www.asd.org/Projects/AprosModel1/BaseRealization
 *    - http://www.asd.org/Projects/AprosModel1/Experiment/UUID
 *  Relative Variable Identifier (RVI):
 *   A reference which can be used to obtain a variable when realization is given 
 *   Examples:
 *   - /Diagram/PI_X#PI_MASS_FLOW
 *   - /Diagram/PI_X#PI_MASS_FLOW#Unit
 *
 * Use cases:
 * 
 * 1. Given a configuration resource and context variable obtain variable
 *    Procedure: call Variable.browse(ReadGraph, Resource)
 *
 * 2. Given two variables obtain RVI of second based on first
 *    Procedure: call Variables.getRVI(Variable, Variable)
 *    
 * 3. Given Realization URI obtain Variable
 *    Procedure: call ReadGraph.getResource and then adapt(Resource, Variable.class)
 *    
 * 4. Given Realization URI and RVI, obtain Variable
 *    Procedure: obtain Variable from Realization URI, call Variable.browse(ReadGraph, String)
 * 
 * 5. Given Variable URI, obtain Variable
 *    Procedure: call Variables.getVariable(ReadGraph, String)
 *     
 * 6. Given Variable, obtain model
 *    Procedure: call Variables.getModel(ReadGraph, Variable)
 *
 * 7. Given Variable, obtain realization
 *    Procedure: call Variables.getRealization(ReadGraph, Variable)

 * 8. Given Variable, obtain Variable in other realization
 *    Procedure: call Variables.switchRealization(ReadGraph, Variable, Resource)
 *    
 */


public interface Variable {

	/*
	 * Convenience method for getting the value of a named property.
	 */
	<T> T getPropertyValue(ReadGraph graph, String name) throws DatabaseException;
	<T> T getPropertyValue(ReadGraph graph, Resource property) throws DatabaseException;
	<T> T getPossiblePropertyValue(ReadGraph graph, String name) throws DatabaseException;
	<T> T getPossiblePropertyValue(ReadGraph graph, Resource property) throws DatabaseException;
	/*
	 * Convenience method for getting the value of a named property.
	 */
	<T> T getPropertyValue(ReadGraph graph, String name, Binding binding) throws DatabaseException;
	<T> T getPropertyValue(ReadGraph graph, Resource property, Binding binding) throws DatabaseException;
	<T> T getPossiblePropertyValue(ReadGraph graph, String name, Binding binding) throws DatabaseException;
	<T> T getPossiblePropertyValue(ReadGraph graph, Resource property, Binding binding) throws DatabaseException;
	/*
	 * Gets the value of the property. Binding is default.
	 */
	<T> T getValue(ReadGraph graph) throws DatabaseException;
	<T> T getPossibleValue(ReadGraph graph) throws DatabaseException;
	/*
	 * Gets the value of the property. Binding is explicitly given.
	 */
	<T> T getValue(ReadGraph graph, Binding binding) throws DatabaseException;
	<T> T getPossibleValue(ReadGraph graph, Binding binding) throws DatabaseException;
    Variant getVariantValue(ReadGraph graph) throws DatabaseException;
	
    /*
     * If role is Property returns the corresponding PropertyInfo.
     * @throws DatabaseException if role of this variable is not Property
     */
    PropertyInfo getPropertyInfo(ReadGraph graph) throws DatabaseException;

    /**
     * @return the index root resource or <code>null</code> if the index root
     *         cannot be found
     * @throws DatabaseException in any other error conditions
     */
    Resource getIndexRoot(ReadGraph graph) throws DatabaseException;

	/**
	 * Writes a value using the given binding.
	 * 
	 * @throws NonWritableVariableException if the variable is not writable
	 * @throws DatabaseException in any other error conditions 
	 */
	void setValue(WriteGraph graph, Object value, Binding binding) throws DatabaseException;
	/**
	 * Writes a value with default binding based on the value class.
	 * 
	 * @throws NonWritableVariableException if the variable is not writable
	 * @throws DatabaseException in any other error conditions 
	 */
	void setValue(WriteGraph graph, Object value) throws DatabaseException;

	/**
	 * Writes a value to the given property using the given binding.
	 * 
	 * @throws NonWritableVariableException if the variable is not writable
	 * @throws DatabaseException in any other error conditions 
	 */
	void setPropertyValue(WriteGraph graph, String name, Object value, Binding binding) throws DatabaseException;
	void setPropertyValue(WriteGraph graph, Resource property, Object value, Binding binding) throws DatabaseException;
	/**
	 * Writes a value to the given property using the default binding based on
	 * the value class.
	 * 
	 * @throws NonWritableVariableException if the variable is not writable
	 * @throws DatabaseException in any other error conditions 
	 */
	void setPropertyValue(WriteGraph graph, String name, Object value) throws DatabaseException;
	void setPropertyValue(WriteGraph graph, Resource property, Object value) throws DatabaseException;

	// TODO methods?

	/*
	 * Gets a named child of this variable. A child corresponds to a '/' in URI notation
	 */
	Variable getChild(ReadGraph graph, String name) throws DatabaseException;
	Variable getPossibleChild(ReadGraph graph, String name) throws DatabaseException;
	/*
	 * Gets a named property of this variable. A property corresponds to a '#' in URI notation
	 */
	Variable getProperty(ReadGraph graph, String name) throws DatabaseException;	
	Variable getProperty(ReadGraph graph, Resource property) throws DatabaseException;	
	Variable getPossibleProperty(ReadGraph graph, String name) throws DatabaseException;	
	Variable getPossibleProperty(ReadGraph graph, Resource property) throws DatabaseException;	

	/*
	 * Browses all children of this variable.
	 * @deprecated replaced by {@link #getChildren(ReadGraph)}
	 */
	@Deprecated
	Collection<Variable> browseChildren(ReadGraph graph) throws DatabaseException;

	/*
	 * Gets all children of this variable.
	 */
	Collection<Variable> getChildren(ReadGraph graph) throws DatabaseException;

	/*
	 * Browses all available properties of this variable. An empty collection shall
	 * be returned if there are no properties for this variable.
	 * @deprecated replaced by {@link #getProperties(ReadGraph)}
	 */
	@Deprecated
	Collection<Variable> browseProperties(ReadGraph graph) throws DatabaseException;

	/*
	 * Gets all available properties of this variable. An empty collection shall
	 * be returned if there are no properties for this variable.
	 */
	Collection<Variable> getProperties(ReadGraph graph) throws DatabaseException;
	/*
	 * Gets all properties of this variable such that {@link #getClassifications()}
	 * contains classification. An empty collection shall
	 * be returned if there are no properties for this variable.
	 */
	Collection<Variable> getProperties(ReadGraph graph, String classification) throws DatabaseException;
	Collection<Variable> getProperties(ReadGraph graph, Resource classification) throws DatabaseException;

	/*
	 * Browses a single variable using the given suffix path. The suffix can contain various '/' and '#'. 
	 */
	Variable browse(ReadGraph graph, String suffix) throws DatabaseException;
	Variable browsePossible(ReadGraph graph, String suffix) throws DatabaseException;

	/*
	 * Browses a single variable using the given configuration resource. 
	 */
	Variable browse(ReadGraph graph, Resource config) throws DatabaseException;
	Variable browsePossible(ReadGraph graph, Resource config) throws DatabaseException;

	Variable resolve(ReadGraph graph, RVIPart part) throws DatabaseException;
	Variable resolvePossible(ReadGraph graph, RVIPart part) throws DatabaseException;

	/*
	 * Gets the requested interface associated to this variable if available.
	 * Examples: 
	 * -Resource
	 * -Accessor
	 * -Binding
	 * 
	 */
	@Deprecated
	<T> T getInterface(ReadGraph graph, Class<T> clazz) throws DatabaseException;

	<T> T adapt(ReadGraph graph, Class<T> clazz) throws DatabaseException;
	<T> T adaptPossible(ReadGraph graph, Class<T> clazz) throws DatabaseException;

	/*
	 * Gets the unique URI reference for this variable.
	 */
	String getURI(ReadGraph graph) throws DatabaseException;

	/*
	 * Standard properties
	 */

	/**
	 * @param graph
	 * @return the unescaped name of this variable. Escaped names are used in URIs.
	 * @throws DatabaseException if no name is available or other database error occurs
	 */
	String getName(ReadGraph graph) throws DatabaseException;

	/**
	 * Informative label of this variable.
	 * 
	 * @param graph
	 * @return
	 * @throws DatabaseException if no label is available or other database error occurs
	 */
	String getLabel(ReadGraph graph) throws DatabaseException;
	/**
	 * @param graph
	 * @return parent variable, <code>null</code> for root variable
	 * @throws DatabaseException
	 */
	Variable getParent(ReadGraph graph) throws DatabaseException;

	Datatype getDatatype(ReadGraph graph) throws DatabaseException;
	Datatype getPossibleDatatype(ReadGraph graph) throws DatabaseException;
	
	Role getRole(ReadGraph graph) throws DatabaseException;
	Role getPossibleRole(ReadGraph graph) throws DatabaseException;

	@Deprecated
	Variable getPredicate(ReadGraph graph) throws DatabaseException;
	@Deprecated
	Variable getPossiblePredicate(ReadGraph graph) throws DatabaseException;

	Resource getPredicateResource(ReadGraph graph) throws DatabaseException;
	Resource getPossiblePredicateResource(ReadGraph graph) throws DatabaseException;

	Resource getRepresents(ReadGraph graph) throws DatabaseException;
	Resource getPossibleRepresents(ReadGraph graph) throws DatabaseException;

	Resource getType(ReadGraph graph) throws DatabaseException;
	Resource getPossibleType(ReadGraph graph) throws DatabaseException;
	Resource getType(ReadGraph graph, Resource baseType) throws DatabaseException;
	Resource getPossibleType(ReadGraph graph, Resource baseType) throws DatabaseException;

	Set<String> getClassifications(ReadGraph graph) throws DatabaseException;

	RVI getRVI(ReadGraph graph) throws DatabaseException;
	RVI getPossibleRVI(ReadGraph graph) throws DatabaseException;

}
