/*******************************************************************************
 * Copyright (c) 2014 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.adapter.impl;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.StringModifier;
import org.simantics.layer0.Layer0;

/**
 * A modifier that enforces all the same naming validations as
 * {@link EntityNameModifier} but also makes sure that entities directly related
 * to the specified entity with the specified relation are renamed at the same
 * time.
 * 
 * @author Tuukka Lehtonen
 * @see EntityNameModifier
 */
public final class RelatedEntityNameModifier implements StringModifier {

	private EntityNameModifier entityModifier;
	private Resource entity;
	private Resource relation;

	public RelatedEntityNameModifier(ReadGraph graph, Resource entity, Resource property, Resource relation) throws DatabaseException {
		this.entityModifier = new EntityNameModifier(graph, entity, property);
		this.entity = entity;
		this.relation = relation;
	}

	@Override
	public String getValue() {
		return entityModifier.getValue();
	}

	@Override
	public String isValid(String value) {
		return entityModifier.isValid(value);
	}

	@Override
	final public void modify(WriteGraph graph, String value) throws DatabaseException {
		entityModifier.modify(graph, value);

		Layer0 L0 = Layer0.getInstance(graph);
		String v = entityModifier.finalValue(value); 
		for (Resource related : graph.getObjects(entity, relation))
			graph.claimLiteral(related, L0.HasName, L0.NameOf, v, Bindings.STRING);
	}

}
