/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.example;

import java.util.ArrayList;
import java.util.Arrays;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.IntegerBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.impl.IntegerBindingDefault;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.IntegerType;

public class BindingExample {

	@SuppressWarnings("unchecked")
	public static void main(String[] args) throws Exception {

		//
		// Binding is an object that makes modifications and conversions  
		// between Datavalues and Java Objects possible.
        //
		//
		// There are 13 datatypes. There is an abstract binding base class 
		// for each datatype, and a concrete class for each java class mapping.
        //
		// For example, IntegerBinding binds to Integer datatype. It is abstract
		// class and does not bind to any specific java class.
		//  
		// It's sub-classes IntegerJavaBinding and MutableIntegerBinding bind to
		// specific classes (java.lang.Integer and MutableInteger).
		//

		IntegerBinding binding = Bindings.INTEGER;
		
		// We can create a java object with a binding
		Object obj = binding.create( 5 /* A valid value of Integer Datatype */ );
		
		// We know that the binding class binds java.lang.Integer, we can typecast 
		Integer number = (Integer) obj;
		
		// We can read the value in a Java Object into a integer value
		int value = binding.getValue_( obj );
				
		// The object can be cloned. Immutable classes are recycled  
		Object obj2 = binding.clone( obj );

		
		
		// 
		// All Binding classes are comparators, and all values can be ordered. 
		// The compare algorithm is specified here http://dev.simantics.org/index.php/Databoard_Specification#Order
		//
		System.out.println("Sorting array");
		Integer[] array = new Integer[] {5, 2, 10, 6000, -30};
		System.out.println(" unsorted = "+Arrays.toString(array));
		Arrays.sort(array, binding);
		System.out.println( Arrays.toString( array ) );
		System.out.println(" sorted = "+Arrays.toString(array));
		
		
		//
		// Bindings can do hashCode and equals aswell
		//
		binding.hashValue( 5 );
		binding.equals(5, 6);
		
		
		// 
		// Data types can have restrictions. Bindings can validate that datavalues 
		// are valid according to the restrictions of the type.
		//
		binding = new IntegerBindingDefault( new IntegerType(null, "[0..10]") );
		try {
			value = 0;
			System.out.print("Validating "+value+" ");
			binding.assertInstaceIsValid(value);
			System.out.println("ok");
		} catch (BindingException e) {
			System.err.println(e);
		}
		try {
			value = 11;
			System.out.print("Validating "+value+" ");
			binding.assertInstaceIsValid(value);
		} catch (BindingException e) {
			System.out.println(e);
		}
		System.out.println();
		
		
		//
		// Bindings can be acquired from class description
		//
		Binding b1 = Bindings.getBinding( int[].class );
		Binding b2 = Bindings.getBinding( Integer[].class );
		
		// 
		// Bindings can be acquired from a datatype
		//
		Datatype arrayType = new ArrayType( Datatypes.INTEGER ); /* Integer[] */				
		Binding b3 = Bindings.getMutableBinding( arrayType ); /* ArrayList<Integer> */ 
		
		
		//
		// Objects can be converted between bindings  
		// 		
		int[] array1 = new int[] {1, 10, 100, 1000, -23432};
		Integer[] array2 = (Integer[]) Bindings.adapt(array1, b1, b2);
		ArrayList<Integer> array3 = (ArrayList<Integer>) Bindings.adapt(array2, b2, b3);
		
		System.out.println( "Adapting classes:");
		System.out.println( " int[]              = "+Arrays.toString(array1) );
		System.out.println( " Integer[]          = "+Arrays.toString(array2) );
		System.out.println( " ArrayList<Integer> = "+array3 );
		System.out.println();
		
		
		
		//
		// Binding can create a default value for any data type.
		// Default value is specified at http://dev.simantics.org/index.php/index.php/Databoard_Specification#Default_Value 
		//		
		value = (Integer) binding.createDefault();
		
		// Default value is always valid.
		Datatype type = new IntegerType(null, "[100..200]");
		binding = new IntegerBindingDefault( (IntegerType) type );
		value = (Integer) binding.createDefault();
		System.out.println("Default Value for value for "+type+" is "+value);
		
		//
		// Binding can create a valid random value
		//
		value = (Integer) binding.createRandom(23243);
		System.out.println("Random Value for value for "+type+" is "+value);
		
		
		// 
		// Values can be serialized to byte[] and back to java Objects
		// Binary serialization format is specified here, 
		// http://dev.simantics.org/index.php/Databoard_Specification#Binary_Serialization_Format
		//
		Serializer s = Bindings.getSerializer( binding );
		value = 50;
		byte[] data = s.serialize( value );
		value = (Integer) s.deserialize( data );
		
		
		//
		// Values can be converted to ascii notation and back
		// Ascii notation is specified at http://dev.simantics.org/index.php/index.php/Databoard_Specification#Datatypes
		//
				
		
	}
	
}

