/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.compressions;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;

/**
 * @author Tuukka Lehtonen
 */
public interface CompressionCodec {

    /**
     * @param inputSize size of uncompressed input data in bytes
     * @return maximum amount of bytes needed for the compressed data
     */
    int compressBound(int inputSize);

    int compress(byte[] uncompressedData, int i, int length,
            byte[] compressedData, int j);
    
    /**
     * Compress a block of data in the input buffer and returns the size of
     * compressed block. The size of input buffer is specified by length. The
     * minimum input buffer size is 16.
     * 
     * <p>
     * The output buffer size must be at least what is reported y
     * {@link #compressBound(int)} for the input data size.
     * 
     * <p>
     * If the input is not compressible, the return value might be larger than
     * length (input buffer size).
     * 
     * <p>
     * The input buffer and the output buffer can not overlap.
     * 
     * <p>
     * It is recommended to have both input buffers as direct or heap buffers,
     * not mixed. Mixing different types of buffers will hurt performance a lot.
     * If both buffers are direct byte buffers and native decompression is
     * available, it will be employed.
     */
    int compressBuffer(ByteBuffer input, int inputOffset, int length,
            ByteBuffer output, int outputOffset);

    byte[] decompress(byte[] compressedData, int i, int uncompressedLength);
    
    /**
     * Decompress a block of compressed data and returns the size of the
     * decompressed block. If error occurs, e.g. the compressed data is
     * corrupted or the output buffer is not large enough, then 0 (zero) will be
     * returned instead.
     * 
     * <p>
     * The input buffer and the output buffer can not overlap.
     * 
     * <p>
     * Decompression is memory safe and guaranteed not to write the output
     * buffer more than what is specified in maxout.
     * 
     * <p>
     * It is recommended to have both input buffers as direct or heap buffers,
     * not mixed. Mixing different types of buffers will hurt performance a lot.
     * If both buffers are direct byte buffers and native decompression is
     * available, it will be employed.
     */
    int decompressBuffer(ByteBuffer input, int inputOffset, int length,
            ByteBuffer output, int outputOffset, int maxout);

    /**
     * @param file
     *            the compressed file to read
     * @return input stream that decompresses its output using the codec's
     *         algorithm
     * @throws FileNotFoundException
     *             see {@link FileOutputStream#FileOutputStream(File)} for when
     *             this is thrown
     */
    InputStream read(File file) throws FileNotFoundException;

    /**
     * @param file
     *            the compressed file to write
     * @return output stream that compresses its input using the codec's
     *         algorithm
     * @throws FileNotFoundException
     *             see {@link FileOutputStream#FileOutputStream(File)} for when
     *             this is thrown
     */
    OutputStream write(File file) throws FileNotFoundException;

    String getId();

}