package org.simantics.charts.query;

import java.util.Arrays;

import org.simantics.charts.ontology.ChartResource;
import org.simantics.charts.ui.ChartData;
import org.simantics.charts.ui.ChartData.ItemKey;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.trend.configuration.TrendItem;
import org.simantics.trend.configuration.YAxisMode;

public class ChartDataQuery extends ResourceRead<ChartData> {

    public ChartDataQuery(Resource resource) {
        super(resource);
    }

    @Override
    public ChartData perform(ReadGraph g) throws DatabaseException {
        Resource chart = resource;
        ChartData data = new ChartData();

        Layer0 L0 = Layer0.getInstance(g);
        ChartResource CHART = ChartResource.getInstance(g);

        data.name = g.getPossibleRelatedValue(chart, L0.HasName, Bindings.STRING);

        Resource am = g.getPossibleObject(chart, CHART.Chart_YAxisMode);
        if ( CHART.YAxisMode_SingleAxis.equals(am) ) data.axisMode = YAxisMode.SingleAxis;
        else if ( CHART.YAxisMode_MultiAxis.equals(am) ) data.axisMode = YAxisMode.MultiAxis;
        else {
            data.axisMode = YAxisMode.SingleAxis;
        }

        Double increment = g.getPossibleRelatedAdapter(chart, CHART.Chart_TimeWindowIncrement, Double.class);
        if ( increment!=null ) data.timeIncrement = increment;
        Double length = g.getPossibleRelatedAdapter(chart, CHART.Chart_TimeWindowLength, Double.class);
        if ( length !=null ) data.timeLength = length;
        Double start = g.getPossibleRelatedAdapter(chart, CHART.Chart_TimeWindowStart, Double.class);
        if ( start!=null ) data.timeStart = start;

        Boolean showMilestones = g.getPossibleRelatedValue(chart, CHART.Chart_ShowMilestones, Bindings.BOOLEAN);
        if ( showMilestones!=null ) data.showMilestones = showMilestones;
        Boolean showGrid = g.getPossibleRelatedValue(chart, CHART.Chart_showGrid, Bindings.BOOLEAN);
        if ( showGrid!=null ) data.showGrid = showGrid;
        Boolean trackExperimentTime = g.getPossibleRelatedValue(chart, CHART.Chart_trackExperimentTime, Bindings.BOOLEAN);
        if ( trackExperimentTime!=null ) data.trackExperimentTime = trackExperimentTime;

        for (Resource chartItem : g.syncRequest(new ObjectsWithType(chart, L0.ConsistsOf, CHART.Chart_Item))) {
            TrendItem ti = g.syncRequest(new TrendItemQuery(chartItem));
            NamedResource nr = new NamedResource(ti.label, chartItem);
            ItemKey key = new ItemKey(ti.renderer, ti.index, nr);
            data.allItems.put(key, ti);
            Boolean hidden = g.getPossibleRelatedValue(chartItem, CHART.Chart_Item_hidden, Bindings.BOOLEAN);
            if (Boolean.TRUE.equals(hidden))
                data.hiddenItems.add(key);
        }

        float[] bgColor = g.getPossibleRelatedValue(chart, CHART.Chart_backgroundColor, Bindings.FLOAT_ARRAY);
        if (bgColor != null && bgColor.length >= 3) {
            int len = bgColor.length;
            if (len >= 6) {
                data.backgroundGradient = true;
                data.backgroundColor1 = Arrays.copyOfRange(bgColor, 0, 3);
                data.backgroundColor2 = Arrays.copyOfRange(bgColor, 3, 6);
            } else if (len >= 3) {
                data.backgroundGradient = false;
                data.backgroundColor1 = Arrays.copyOfRange(bgColor, 0, 3);
                data.backgroundColor2 = data.backgroundColor1;
            }
        }

        float[] gridColor = g.getPossibleRelatedValue(chart, CHART.Chart_gridColor, Bindings.FLOAT_ARRAY);
        if (gridColor != null && gridColor.length >= 3) {
            data.gridColor = gridColor;
        }

        return data;
    }

}
