/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.simantics.browsing.ui.BuiltinKeys.ViewpointKey;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.property.IProperty;
import org.simantics.browsing.ui.content.Viewpoint;
import org.simantics.browsing.ui.content.ViewpointFactory;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.RelationContextImpl;
import org.simantics.layer0.Layer0;

/**
 * A basic viewpoint which follows the Has Property relation to show properties
 * of resources.
 * 
 * <p>
 * The viewpoint generates {@link ResourceArrayProperty} output for Value instances that contain array values.
 * </p>
 * 
 * @see ArrayPropertyValueViewpointFactory viewpoint with IArrayProperty support
 */
public class PropertyViewpointFactory implements ViewpointFactory {
    @Override
    public Viewpoint create(PrimitiveQueryUpdater provider, final NodeContext context, ViewpointKey key) {
        return new LazyViewpoint(provider, context, key) {
            @Override
            public NodeContext[] children(ReadGraph graph) throws DatabaseException {
                Object in = getInput(Object.class);
                Resource input = null;
                if (in instanceof Resource) {
                    input = (Resource) in;
                } else if (in instanceof ResourceArray) {
                    ResourceArray ra = (ResourceArray) in;
                    if (ra.isEmpty())
                        return NodeContext.NONE;
                    input = ra.resources[0];
                } else if (in instanceof Statement) {
                    input = ((Statement) in).getObject();
                } else {
                    return NodeContext.NONE;
                }

//                // First see if this is a Value resource.
//                NodeContext[] valueChildren = GraphPropertyUtil.tryValueGetChildren(graph, input, new ResourcePropertyFactory(IProperty.Type.DIRECT, input));
//                if (valueChildren != null)
//                    return valueChildren;

                // If not, then go look for properties, either direct or asserted.
                Collection<Object> properties = new ArrayList<Object>();
                Set<Object> usedProperties = new HashSet<Object>();
                Layer0 L0 = Layer0.getInstance(graph);
                for (Resource predicate : graph.getPredicates(input)) {
                    if (!graph.isSubrelationOf(predicate, L0.IsRelatedTo))
                        continue;

                    for (Statement stm : graph.getStatements(input, predicate)) {
                        Resource object = stm.getObject();
                        //System.out.println("STM: " + NameUtils.toString(graph, stm));
                        if (usedProperties.add(stm.getPredicate())) {
                            if (
                                    // [TLe] Added to keep data that can't be
                                    // visualized anyway out of the basic property view.
                                    graph.isInstanceOf(stm.getObject(), L0.Literal)
                                    && (
                                           graph.isSubrelationOf(stm.getPredicate(), L0.HasProperty)
                                           || graph.syncRequest(new IsEnumeratedValue(object))
                                           )
                                    )
                            {
                                IProperty.Type type = stm.getSubject().equals(input) ? IProperty.Type.DIRECT : IProperty.Type.ASSERTED;
                                //System.out.println("\tTYPE: " + type);
                                IProperty prop = GraphPropertyUtil.tryCreateProperty(graph, object, new ResourcePropertyFactory(type, input, stm.getPredicate(), object, ResourceArray.EMPTY));
                                //System.out.println("\tPROP: " + prop);
                                if (prop != null)
                                    properties.add(prop);
                                else
                                    properties.add(new RelationContextImpl(input, stm));
                            }
                        }
                    }
                }

                return toContextsWithInput(properties);
            }

            @Override
            public Boolean hasChildren(ReadGraph graph) throws DatabaseException {
                return children(graph).length > 0;
            }
        };
    }

    @Override
    public String toString() {
        return "Properties";
    }

}
