/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.viewpoints.ViewpointStub;
import org.simantics.browsing.ui.content.Viewpoint;
import org.simantics.browsing.ui.graph.impl.request.ResourceQuery;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.datastructures.Pair;

/**
 * Implement {@link #children(ReadGraph)} and {@link #hasChildren(ReadGraph)}.
 * 
 * @author Tuukka Lehtonen
 */
public abstract class LazyViewpoint extends ViewpointStub {

    /**
     * Needed for separating childQuery and hasChildQuery from each other in the
     * equals sense.
     */
    private static final Object                 CHILDREN     = new Object();

    final private ResourceQuery<NodeContext[]> childQuery;

    final protected PrimitiveQueryUpdater       updater;
    final protected NodeContext                context;
    final protected BuiltinKeys.ViewpointKey    key;

    /**
     * @param graph
     * @return
     */
    public abstract NodeContext[] children(ReadGraph graph) throws DatabaseException;

    /**
     * @param graph
     * @return
     */
    public abstract Boolean hasChildren(ReadGraph graph) throws DatabaseException;

    /**
     * This identity is used to give the back-end graph requests a
     * <em>properly unique</em> identity that so that the graph back-end caching
     * and graph explorer node context caching work together properly.
     * 
     * Consider having two graph explorer instances that have the same
     * configuration (same evaluators) and are showing the same resource from
     * the graph database. In this case the requests are actually meant to have
     * an identical identity and performing the graph request will simply bind a
     * new listener for the one and same request.
     * 
     * @return an additional identity for graph back-end requests to make them
     *         properly unique
     */
    public Object getIdentity() {
        return key;
    }

    public LazyViewpoint(final PrimitiveQueryUpdater updater, NodeContext context, BuiltinKeys.ViewpointKey key) {

        assert updater != null;
        assert context != null;
        assert key != null;

        this.updater = updater;
        this.context = context;
        this.key = key;

        this.childQuery = new ResourceQuery<NodeContext[]>(Pair.make(getIdentity(), CHILDREN), context) {

            @Override
            public NodeContext[] perform(ReadGraph graph) throws DatabaseException {
                return children(graph);
            }

            @Override
            public String toString() {
                return LazyViewpoint.this.toString() + "[CHILDREN]";
            }

        };

    }

    private Listener<NodeContext[]> createListener() {
    	
    	return new Listener<NodeContext[]>() {

    		boolean executed = false;
    		boolean disposed = false;
    		
            @Override
            public void execute(NodeContext[] result) {
                replaceChildrenResult(result);
                executed = true;
            }

            @Override
            public boolean isDisposed() {
             	if(disposed) return true;

            	if((updater.isDisposed() || !updater.isShown(context)) && executed) {
            		children = Viewpoint.PENDING_CHILDREN;
            		disposed = true;
            		return true;
            	} else {
            		return false;
            	}
            	
            }

            public void exception(Throwable t) {
                System.out.print("LazyViewpoint.childQuery failed: ");
                t.printStackTrace();
            }

            @Override
            public String toString() {
                return "LazyViewpoint[" + System.identityHashCode(LazyViewpoint.this) + "].childProcedure";
            }

        };
    	
    }
    
    public NodeContext getContext() {
        return context;
    }

    @Override
    public NodeContext[] getChildren() {

        if (children == Viewpoint.PENDING_CHILDREN) {
            DataSource<AsyncReadGraph> source = updater.getDataSource(AsyncReadGraph.class);
            final Listener<NodeContext[]> childProcedure = createListener();
            source.schedule(graph -> graph.asyncRequest(childQuery, childProcedure));
        }

        return children;

    }

    @Override
    public Boolean getHasChildren() {
    	return getChildren().length > 0;
    }

    protected void replaceChildrenResult(NodeContext[] result) {
        setChildren(updater, result);
        updater.scheduleReplace(context, key, this);
    }

    /**
     * @param <T>
     * @param clazz
     * @return input of the specified class
     * @throws ClassCastException if the input class does not match the
     *         specified class
     * @throws NullPointerException if the input is null
     */
    @SuppressWarnings("unchecked")
    protected <T> T getInput(Class<T> clazz) throws ClassCastException {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o == null)
            throw new NullPointerException("null input");
        return (T) o;
    }

    /**
     * @param <T>
     * @param clazz
     * @return <code>null</code> if input is <code>null</code> or if the class does not match
     */
    @SuppressWarnings("unchecked")
    protected <T> T tryGetInput(Class<T> clazz) {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o != null && clazz.isInstance(o))
            return (T) o;
        return null;
    }

}
