/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.extension.internal;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.simantics.browsing.ui.common.Activator;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.content.Contributor;
import org.simantics.utils.strings.StringUtils;

/**
 * @author Tuukka Lehtonen
 */
public abstract class ContributorExtensionManager<Factory> implements IExtensionChangeHandler {

    private final static String                NAMESPACE    = Activator.PLUGIN_ID;

    private ExtensionTracker                   tracker;

    @SuppressWarnings("unchecked")
    ContributorExtension<Factory>[]            extensions   = new ContributorExtension[0];

    abstract public String getExtensionPointName();

    ContributorExtensionManager() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, getExtensionPointName());
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    @SuppressWarnings("unchecked")
    void close() {
        tracker.close();
        tracker = null;
        extensions = new ContributorExtension[0];
    }

    public ContributorExtension<Factory>[] getExtensions() {
        return extensions;
    }

    @SuppressWarnings("unchecked")
    private void loadExtensions(IConfigurationElement[] elements) {

        Set<ContributorExtension<Factory>> newExtensions = new HashSet<ContributorExtension<Factory>>(Arrays.asList(extensions));

        for (IConfigurationElement el : elements) {
            try {

                String id = StringUtils.safeString(el.getAttribute("id"));
                Contributor<Factory> factory = (Contributor<Factory>)el.createExecutableExtension("class");
                ContributorExtension<Factory> ext = new ContributorExtensionImpl<Factory>(id, factory);

                // Start tracking the new extension object, its removal will be notified of
                // with removeExtension(extension, Object[]).
                tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

                newExtensions.add(ext);
            } catch (CoreException e) {
                ErrorLogger.defaultLogError("Failed to initialize " + getExtensionPointName() + " extension \"" + el.getName()
                        + "\" with name \"" + el.getAttribute("name") + "\": "
                        + e.getMessage(), e);
            }
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new ContributorExtension[newExtensions.size()]);
    }

    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @SuppressWarnings("unchecked")
    @Override
    public void removeExtension(IExtension extension, Object[] objects) {
        Set<ContributorExtension<Factory>> newExtensions = new HashSet<ContributorExtension<Factory>>(Arrays.asList(extensions));

        for (Object o : objects) {
            tracker.unregisterObject(extension, o);
            newExtensions.remove(o);
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new ContributorExtension[newExtensions.size()]);
    }

}
