/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.simantics.db.layer0.exception.InvalidVariableException;


/**
 * ErrorLogger sends error messages to "Error log" view.
 * 
 * @author Toni Kalajainen
 */
public class ErrorLogger {

    public static final boolean ECHO = true;

    private final Plugin plugin;
    private String pluginID;

    public ErrorLogger(Plugin plugin) {
        this.plugin = plugin;
    }

    private String getPluginID() {
        if (pluginID==null)
            pluginID = plugin.getBundle().getSymbolicName();
        return pluginID;
    }

    /**
     * Log a warning event.
     * 
     * @param message message of the error
     * @param exception the exception, or <code>null</code>
     */
    public void logWarning(String message, Throwable exception) {
        log(IStatus.WARNING, IStatus.OK, message, exception);
    }

    /**
     * Log a message.
     * 
     * @param message message of the error
     * @param exception the exception, or <code>null</code>
     */
    public void logMessage(String message, Throwable exception) {
        log(IStatus.INFO, IStatus.OK, message, exception);
    }

    /**
     * Log an error event.
     * 
     * @param message message of the error
     * @param exception the exception, or <code>null</code>
     */
    public void logError(String message, Throwable exception) {
        // Errors are much more useful with a stack trace!
        if (exception == null) {
            exception = new RuntimeException();
        }
        log(IStatus.ERROR, IStatus.OK, message, exception);
    }

    public void log(int severity, int code, String message, Throwable exception) {
        IStatus status = new Status(severity, getPluginID(), code, message, exception);
        log(status);
    }

    public void log(IStatus status) {
        plugin.getLog().log(status);
    }

    public static ErrorLogger getDefault() {
        return Activator.getDefault().getErrorLogger();
    }

    public static void defaultLogError(String message, Throwable exception) {
        if (filterException(exception))
            return;
        getDefault().logError(message, exception);
        if(ECHO && exception != null) exception.printStackTrace();
    }

    public static void defaultLogError(Throwable exception) {
        if (filterException(exception))
            return;
        getDefault().logError(getUIFriendErrorMessage(exception), exception);
        if(ECHO && exception != null) exception.printStackTrace();
    }

    public static void defaultLogWarning(String message, Throwable exception) {
        if (filterException(exception))
            return;
        getDefault().logWarning(message, exception);
        if(ECHO && exception != null) exception.printStackTrace();
    }

    public static void defaultLogWarning(Throwable exception) {
        if (filterException(exception))
            return;
        getDefault().logWarning(getUIFriendErrorMessage(exception), exception);
        if(ECHO && exception != null) exception.printStackTrace();
    }

    public static void defaultLog(IStatus status) {
        getDefault().log(status);
        if(ECHO) System.out.println(status);
    }

    /**
     * This call makes verbose error message that is suitable for
     * UI Dialgos. The full cause hierarchy is shown with the tree.
     * In this case verbose includes class names.
     * 
     * @param e exception
     * @return message
     */
    public static String getUIFriendErrorMessageVerbose(Throwable e) {
        String result = "";
        Throwable pe = null;
        while (e!=null && pe!=e) {
            result += e.getClass().getName()+": "+e.getMessage()+"\n";
            pe = e;
            e = e.getCause();
        }
        return result;
    }

    /**
     * This call makes error message that is suitable for
     * UI Dialogs. The full cause hierarchy is shown with the tree.
     * In this case verbose includes class names.
     * 
     * @param e exception
     * @return message
     */
    public static String getUIFriendErrorMessage(Throwable e) {
        String result = "";
        Throwable pe = null;
        while (e!=null && pe!=e) {
            result += e.getMessage()+"\n";
            pe = e;
            e = e.getCause();
        }
        return result;
    }

    public static boolean filterException(Throwable t) {
        if (t instanceof InvalidVariableException)
            return true;
        return false;
    }

}
